package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link InterfaceStateChangeBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     InterfaceStateChangeBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new InterfaceStateChangeBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of InterfaceStateChangeBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see InterfaceStateChangeBuilder
 * @see Builder
 *
 */
public class InterfaceStateChangeBuilder implements Builder<InterfaceStateChange> {

    private InterfaceStatus _adminStatus;
    private InterfaceNameOrIndex _name;
    private InterfaceStatus _operStatus;


    Map<Class<? extends Augmentation<InterfaceStateChange>>, Augmentation<InterfaceStateChange>> augmentation = Collections.emptyMap();

    public InterfaceStateChangeBuilder() {
    }

    public InterfaceStateChangeBuilder(InterfaceStateChange base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<InterfaceStateChange>>, Augmentation<InterfaceStateChange>> aug =((AugmentationHolder<InterfaceStateChange>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._adminStatus = base.getAdminStatus();
        this._name = base.getName();
        this._operStatus = base.getOperStatus();
    }


    public InterfaceStatus getAdminStatus() {
        return _adminStatus;
    }
    
    public InterfaceNameOrIndex getName() {
        return _name;
    }
    
    public InterfaceStatus getOperStatus() {
        return _operStatus;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<InterfaceStateChange>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public InterfaceStateChangeBuilder setAdminStatus(final InterfaceStatus value) {
        this._adminStatus = value;
        return this;
    }
    
    public InterfaceStateChangeBuilder setName(final InterfaceNameOrIndex value) {
        this._name = value;
        return this;
    }
    
    public InterfaceStateChangeBuilder setOperStatus(final InterfaceStatus value) {
        this._operStatus = value;
        return this;
    }
    
    public InterfaceStateChangeBuilder addAugmentation(Class<? extends Augmentation<InterfaceStateChange>> augmentationType, Augmentation<InterfaceStateChange> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public InterfaceStateChangeBuilder removeAugmentation(Class<? extends Augmentation<InterfaceStateChange>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public InterfaceStateChange build() {
        return new InterfaceStateChangeImpl(this);
    }

    private static final class InterfaceStateChangeImpl
        extends AbstractAugmentable<InterfaceStateChange>
        implements InterfaceStateChange {
    
        private final InterfaceStatus _adminStatus;
        private final InterfaceNameOrIndex _name;
        private final InterfaceStatus _operStatus;
    
        InterfaceStateChangeImpl(InterfaceStateChangeBuilder base) {
            super(base.augmentation);
            this._adminStatus = base.getAdminStatus();
            this._name = base.getName();
            this._operStatus = base.getOperStatus();
        }
    
        @Override
        public InterfaceStatus getAdminStatus() {
            return _adminStatus;
        }
        
        @Override
        public InterfaceNameOrIndex getName() {
            return _name;
        }
        
        @Override
        public InterfaceStatus getOperStatus() {
            return _operStatus;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_adminStatus);
            result = prime * result + Objects.hashCode(_name);
            result = prime * result + Objects.hashCode(_operStatus);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!InterfaceStateChange.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            InterfaceStateChange other = (InterfaceStateChange)obj;
            if (!Objects.equals(_adminStatus, other.getAdminStatus())) {
                return false;
            }
            if (!Objects.equals(_name, other.getName())) {
                return false;
            }
            if (!Objects.equals(_operStatus, other.getOperStatus())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                InterfaceStateChangeImpl otherImpl = (InterfaceStateChangeImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<InterfaceStateChange>>, Augmentation<InterfaceStateChange>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("InterfaceStateChange");
            CodeHelpers.appendValue(helper, "_adminStatus", _adminStatus);
            CodeHelpers.appendValue(helper, "_name", _name);
            CodeHelpers.appendValue(helper, "_operStatus", _operStatus);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
