package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMap.Builder;
import java.lang.Integer;
import java.lang.Override;
import java.lang.String;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.opendaylight.yangtools.yang.binding.Enumeration;

public enum SpanState implements Enumeration {
    /**
     * Receive (Rx) SPAN—The goal of receive (or ingress) SPAN is to monitor as much as
     * possible all the packets received by the source interface. A copy of each packet
     * received by the source is sent to the destination port for that SPAN session.
     * You can monitor a series or range of ingress ports in a SPAN session.
     *
     */
    Receive(1, "receive"),
    
    /**
     * Transmit (Tx) SPAN—The goal of transmit (or egress) SPAN is to monitor as much
     * as possible all the packets sent by the source interface after all modification
     * and processing is performed by the switch. A copy of each packet sent by the
     * source is sent to the destination port for that SPAN session. The copy is
     * provided after the packet is modified. Monitors a range of egress ports in a
     * SPAN session.
     *
     */
    Transmit(2, "transmit"),
    
    /**
     * In a SPAN session, you can monitor a series or range of ports for both received
     * and sent packets.
     *
     */
    Both(3, "both")
    ;

    private static final Map<String, SpanState> NAME_MAP;
    private static final Map<Integer, SpanState> VALUE_MAP;

    static {
        final Builder<String, SpanState> nb = ImmutableMap.builder();
        final Builder<Integer, SpanState> vb = ImmutableMap.builder();
        for (SpanState enumItem : SpanState.values()) {
            vb.put(enumItem.value, enumItem);
            nb.put(enumItem.name, enumItem);
        }

        NAME_MAP = nb.build();
        VALUE_MAP = vb.build();
    }

    private final String name;
    private final int value;

    private SpanState(int value, String name) {
        this.value = value;
        this.name = name;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public int getIntValue() {
        return value;
    }

    /**
     * Return the enumeration member whose {@link #getName()} matches specified value.
     *
     * @param name YANG assigned name
     * @return corresponding SpanState item, if present
     * @throws NullPointerException if name is null
     */
    public static Optional<SpanState> forName(String name) {
        return Optional.ofNullable(NAME_MAP.get(Objects.requireNonNull(name)));
    }

    /**
     * Return the enumeration member whose {@link #getIntValue()} matches specified value.
     *
     * @param intValue integer value
     * @return corresponding SpanState item, or null if no such item exists
     */
    public static SpanState forValue(int intValue) {
        return VALUE_MAP.get(intValue);
    }
}
