package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607;
import com.google.common.base.MoreObjects;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.Ethernet;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.Gre;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.L2;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.Loopback;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.Routing;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.Span;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.Tap;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.VhostUser;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.Vxlan;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.interfaces._interface.VxlanGpe;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link VppInterfaceAugmentationBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     VppInterfaceAugmentationBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new VppInterfaceAugmentationBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of VppInterfaceAugmentationBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see VppInterfaceAugmentationBuilder
 * @see Builder
 *
 */
public class VppInterfaceAugmentationBuilder implements Builder<VppInterfaceAugmentation> {

    private Ethernet _ethernet;
    private Gre _gre;
    private L2 _l2;
    private Loopback _loopback;
    private Routing _routing;
    private Span _span;
    private Tap _tap;
    private VhostUser _vhostUser;
    private Vxlan _vxlan;
    private VxlanGpe _vxlanGpe;



    public VppInterfaceAugmentationBuilder() {
    }

    public VppInterfaceAugmentationBuilder(VppInterfaceAugmentation base) {
        this._ethernet = base.getEthernet();
        this._gre = base.getGre();
        this._l2 = base.getL2();
        this._loopback = base.getLoopback();
        this._routing = base.getRouting();
        this._span = base.getSpan();
        this._tap = base.getTap();
        this._vhostUser = base.getVhostUser();
        this._vxlan = base.getVxlan();
        this._vxlanGpe = base.getVxlanGpe();
    }


    public Ethernet getEthernet() {
        return _ethernet;
    }
    
    public Gre getGre() {
        return _gre;
    }
    
    public L2 getL2() {
        return _l2;
    }
    
    public Loopback getLoopback() {
        return _loopback;
    }
    
    public Routing getRouting() {
        return _routing;
    }
    
    public Span getSpan() {
        return _span;
    }
    
    public Tap getTap() {
        return _tap;
    }
    
    public VhostUser getVhostUser() {
        return _vhostUser;
    }
    
    public Vxlan getVxlan() {
        return _vxlan;
    }
    
    public VxlanGpe getVxlanGpe() {
        return _vxlanGpe;
    }

    
    public VppInterfaceAugmentationBuilder setEthernet(final Ethernet value) {
        this._ethernet = value;
        return this;
    }
    
    public VppInterfaceAugmentationBuilder setGre(final Gre value) {
        this._gre = value;
        return this;
    }
    
    public VppInterfaceAugmentationBuilder setL2(final L2 value) {
        this._l2 = value;
        return this;
    }
    
    public VppInterfaceAugmentationBuilder setLoopback(final Loopback value) {
        this._loopback = value;
        return this;
    }
    
    public VppInterfaceAugmentationBuilder setRouting(final Routing value) {
        this._routing = value;
        return this;
    }
    
    public VppInterfaceAugmentationBuilder setSpan(final Span value) {
        this._span = value;
        return this;
    }
    
    public VppInterfaceAugmentationBuilder setTap(final Tap value) {
        this._tap = value;
        return this;
    }
    
    public VppInterfaceAugmentationBuilder setVhostUser(final VhostUser value) {
        this._vhostUser = value;
        return this;
    }
    
    public VppInterfaceAugmentationBuilder setVxlan(final Vxlan value) {
        this._vxlan = value;
        return this;
    }
    
    public VppInterfaceAugmentationBuilder setVxlanGpe(final VxlanGpe value) {
        this._vxlanGpe = value;
        return this;
    }
    

    @Override
    public VppInterfaceAugmentation build() {
        return new VppInterfaceAugmentationImpl(this);
    }

    private static final class VppInterfaceAugmentationImpl
        implements VppInterfaceAugmentation {
    
        private final Ethernet _ethernet;
        private final Gre _gre;
        private final L2 _l2;
        private final Loopback _loopback;
        private final Routing _routing;
        private final Span _span;
        private final Tap _tap;
        private final VhostUser _vhostUser;
        private final Vxlan _vxlan;
        private final VxlanGpe _vxlanGpe;
    
        VppInterfaceAugmentationImpl(VppInterfaceAugmentationBuilder base) {
            this._ethernet = base.getEthernet();
            this._gre = base.getGre();
            this._l2 = base.getL2();
            this._loopback = base.getLoopback();
            this._routing = base.getRouting();
            this._span = base.getSpan();
            this._tap = base.getTap();
            this._vhostUser = base.getVhostUser();
            this._vxlan = base.getVxlan();
            this._vxlanGpe = base.getVxlanGpe();
        }
    
        @Override
        public Ethernet getEthernet() {
            return _ethernet;
        }
        
        @Override
        public Gre getGre() {
            return _gre;
        }
        
        @Override
        public L2 getL2() {
            return _l2;
        }
        
        @Override
        public Loopback getLoopback() {
            return _loopback;
        }
        
        @Override
        public Routing getRouting() {
            return _routing;
        }
        
        @Override
        public Span getSpan() {
            return _span;
        }
        
        @Override
        public Tap getTap() {
            return _tap;
        }
        
        @Override
        public VhostUser getVhostUser() {
            return _vhostUser;
        }
        
        @Override
        public Vxlan getVxlan() {
            return _vxlan;
        }
        
        @Override
        public VxlanGpe getVxlanGpe() {
            return _vxlanGpe;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_ethernet);
            result = prime * result + Objects.hashCode(_gre);
            result = prime * result + Objects.hashCode(_l2);
            result = prime * result + Objects.hashCode(_loopback);
            result = prime * result + Objects.hashCode(_routing);
            result = prime * result + Objects.hashCode(_span);
            result = prime * result + Objects.hashCode(_tap);
            result = prime * result + Objects.hashCode(_vhostUser);
            result = prime * result + Objects.hashCode(_vxlan);
            result = prime * result + Objects.hashCode(_vxlanGpe);
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!VppInterfaceAugmentation.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            VppInterfaceAugmentation other = (VppInterfaceAugmentation)obj;
            if (!Objects.equals(_ethernet, other.getEthernet())) {
                return false;
            }
            if (!Objects.equals(_gre, other.getGre())) {
                return false;
            }
            if (!Objects.equals(_l2, other.getL2())) {
                return false;
            }
            if (!Objects.equals(_loopback, other.getLoopback())) {
                return false;
            }
            if (!Objects.equals(_routing, other.getRouting())) {
                return false;
            }
            if (!Objects.equals(_span, other.getSpan())) {
                return false;
            }
            if (!Objects.equals(_tap, other.getTap())) {
                return false;
            }
            if (!Objects.equals(_vhostUser, other.getVhostUser())) {
                return false;
            }
            if (!Objects.equals(_vxlan, other.getVxlan())) {
                return false;
            }
            if (!Objects.equals(_vxlanGpe, other.getVxlanGpe())) {
                return false;
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("VppInterfaceAugmentation");
            CodeHelpers.appendValue(helper, "_ethernet", _ethernet);
            CodeHelpers.appendValue(helper, "_gre", _gre);
            CodeHelpers.appendValue(helper, "_l2", _l2);
            CodeHelpers.appendValue(helper, "_loopback", _loopback);
            CodeHelpers.appendValue(helper, "_routing", _routing);
            CodeHelpers.appendValue(helper, "_span", _span);
            CodeHelpers.appendValue(helper, "_tap", _tap);
            CodeHelpers.appendValue(helper, "_vhostUser", _vhostUser);
            CodeHelpers.appendValue(helper, "_vxlan", _vxlan);
            CodeHelpers.appendValue(helper, "_vxlanGpe", _vxlanGpe);
            return helper.toString();
        }
    }
}
