package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607;
import com.google.common.base.MoreObjects;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.yang.types.rev130715.Counter64;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link VppInterfaceStatisticsAugmentationBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     VppInterfaceStatisticsAugmentationBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new VppInterfaceStatisticsAugmentationBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of VppInterfaceStatisticsAugmentationBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see VppInterfaceStatisticsAugmentationBuilder
 * @see Builder
 *
 */
public class VppInterfaceStatisticsAugmentationBuilder implements Builder<VppInterfaceStatisticsAugmentation> {

    private Counter64 _inErrorsMiss;
    private Counter64 _inErrorsNoBuf;
    private Counter64 _outDiscardsFifoFull;



    public VppInterfaceStatisticsAugmentationBuilder() {
    }

    public VppInterfaceStatisticsAugmentationBuilder(VppInterfaceStatisticsAugmentation base) {
        this._inErrorsMiss = base.getInErrorsMiss();
        this._inErrorsNoBuf = base.getInErrorsNoBuf();
        this._outDiscardsFifoFull = base.getOutDiscardsFifoFull();
    }


    public Counter64 getInErrorsMiss() {
        return _inErrorsMiss;
    }
    
    public Counter64 getInErrorsNoBuf() {
        return _inErrorsNoBuf;
    }
    
    public Counter64 getOutDiscardsFifoFull() {
        return _outDiscardsFifoFull;
    }

    
    public VppInterfaceStatisticsAugmentationBuilder setInErrorsMiss(final Counter64 value) {
        this._inErrorsMiss = value;
        return this;
    }
    
    public VppInterfaceStatisticsAugmentationBuilder setInErrorsNoBuf(final Counter64 value) {
        this._inErrorsNoBuf = value;
        return this;
    }
    
    public VppInterfaceStatisticsAugmentationBuilder setOutDiscardsFifoFull(final Counter64 value) {
        this._outDiscardsFifoFull = value;
        return this;
    }
    

    @Override
    public VppInterfaceStatisticsAugmentation build() {
        return new VppInterfaceStatisticsAugmentationImpl(this);
    }

    private static final class VppInterfaceStatisticsAugmentationImpl
        implements VppInterfaceStatisticsAugmentation {
    
        private final Counter64 _inErrorsMiss;
        private final Counter64 _inErrorsNoBuf;
        private final Counter64 _outDiscardsFifoFull;
    
        VppInterfaceStatisticsAugmentationImpl(VppInterfaceStatisticsAugmentationBuilder base) {
            this._inErrorsMiss = base.getInErrorsMiss();
            this._inErrorsNoBuf = base.getInErrorsNoBuf();
            this._outDiscardsFifoFull = base.getOutDiscardsFifoFull();
        }
    
        @Override
        public Counter64 getInErrorsMiss() {
            return _inErrorsMiss;
        }
        
        @Override
        public Counter64 getInErrorsNoBuf() {
            return _inErrorsNoBuf;
        }
        
        @Override
        public Counter64 getOutDiscardsFifoFull() {
            return _outDiscardsFifoFull;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_inErrorsMiss);
            result = prime * result + Objects.hashCode(_inErrorsNoBuf);
            result = prime * result + Objects.hashCode(_outDiscardsFifoFull);
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!VppInterfaceStatisticsAugmentation.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            VppInterfaceStatisticsAugmentation other = (VppInterfaceStatisticsAugmentation)obj;
            if (!Objects.equals(_inErrorsMiss, other.getInErrorsMiss())) {
                return false;
            }
            if (!Objects.equals(_inErrorsNoBuf, other.getInErrorsNoBuf())) {
                return false;
            }
            if (!Objects.equals(_outDiscardsFifoFull, other.getOutDiscardsFifoFull())) {
                return false;
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("VppInterfaceStatisticsAugmentation");
            CodeHelpers.appendValue(helper, "_inErrorsMiss", _inErrorsMiss);
            CodeHelpers.appendValue(helper, "_inErrorsNoBuf", _inErrorsNoBuf);
            CodeHelpers.appendValue(helper, "_outDiscardsFifoFull", _outDiscardsFifoFull);
            return helper.toString();
        }
    }
}
