package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.bridge.domain.attributes.arp.termination.table;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.IpAddress;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.yang.types.rev130715.PhysAddress;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link ArpTerminationTableEntryBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     ArpTerminationTableEntryBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new ArpTerminationTableEntryBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of ArpTerminationTableEntryBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see ArpTerminationTableEntryBuilder
 * @see Builder
 *
 */
public class ArpTerminationTableEntryBuilder implements Builder<ArpTerminationTableEntry> {

    private IpAddress _ipAddress;
    private PhysAddress _physAddress;
    private ArpTerminationTableEntryKey key;


    Map<Class<? extends Augmentation<ArpTerminationTableEntry>>, Augmentation<ArpTerminationTableEntry>> augmentation = Collections.emptyMap();

    public ArpTerminationTableEntryBuilder() {
    }

    public ArpTerminationTableEntryBuilder(ArpTerminationTableEntry base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<ArpTerminationTableEntry>>, Augmentation<ArpTerminationTableEntry>> aug =((AugmentationHolder<ArpTerminationTableEntry>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._ipAddress = base.getIpAddress();
        this._physAddress = base.getPhysAddress();
    }


    public ArpTerminationTableEntryKey key() {
        return key;
    }
    
    public IpAddress getIpAddress() {
        return _ipAddress;
    }
    
    public PhysAddress getPhysAddress() {
        return _physAddress;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<ArpTerminationTableEntry>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public ArpTerminationTableEntryBuilder withKey(final ArpTerminationTableEntryKey key) {
        this.key = key;
        return this;
    }
    
    public ArpTerminationTableEntryBuilder setIpAddress(final IpAddress value) {
        this._ipAddress = value;
        return this;
    }
    
    public ArpTerminationTableEntryBuilder setPhysAddress(final PhysAddress value) {
        this._physAddress = value;
        return this;
    }
    
    public ArpTerminationTableEntryBuilder addAugmentation(Class<? extends Augmentation<ArpTerminationTableEntry>> augmentationType, Augmentation<ArpTerminationTableEntry> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public ArpTerminationTableEntryBuilder removeAugmentation(Class<? extends Augmentation<ArpTerminationTableEntry>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public ArpTerminationTableEntry build() {
        return new ArpTerminationTableEntryImpl(this);
    }

    private static final class ArpTerminationTableEntryImpl
        extends AbstractAugmentable<ArpTerminationTableEntry>
        implements ArpTerminationTableEntry {
    
        private final IpAddress _ipAddress;
        private final PhysAddress _physAddress;
        private final ArpTerminationTableEntryKey key;
    
        ArpTerminationTableEntryImpl(ArpTerminationTableEntryBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new ArpTerminationTableEntryKey(base.getIpAddress(), base.getPhysAddress());
            }
            this._ipAddress = key.getIpAddress();
            this._physAddress = key.getPhysAddress();
        }
    
        @Override
        public ArpTerminationTableEntryKey key() {
            return key;
        }
        
        @Override
        public IpAddress getIpAddress() {
            return _ipAddress;
        }
        
        @Override
        public PhysAddress getPhysAddress() {
            return _physAddress;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_ipAddress);
            result = prime * result + Objects.hashCode(_physAddress);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!ArpTerminationTableEntry.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            ArpTerminationTableEntry other = (ArpTerminationTableEntry)obj;
            if (!Objects.equals(_ipAddress, other.getIpAddress())) {
                return false;
            }
            if (!Objects.equals(_physAddress, other.getPhysAddress())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                ArpTerminationTableEntryImpl otherImpl = (ArpTerminationTableEntryImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<ArpTerminationTableEntry>>, Augmentation<ArpTerminationTableEntry>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("ArpTerminationTableEntry");
            CodeHelpers.appendValue(helper, "_ipAddress", _ipAddress);
            CodeHelpers.appendValue(helper, "_physAddress", _physAddress);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
