package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.l2.fib.attributes.l2.fib.table;
import com.google.common.base.MoreObjects;
import java.lang.Boolean;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.yang.types.rev130715.PhysAddress;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.L2FibAction;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link L2FibEntryBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     L2FibEntryBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new L2FibEntryBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of L2FibEntryBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see L2FibEntryBuilder
 * @see Builder
 *
 */
public class L2FibEntryBuilder implements Builder<L2FibEntry> {

    private Class<? extends L2FibAction> _action;
    private String _outgoingInterface;
    private PhysAddress _physAddress;
    private Boolean _bridgedVirtualInterface;
    private Boolean _staticConfig;
    private L2FibEntryKey key;


    Map<Class<? extends Augmentation<L2FibEntry>>, Augmentation<L2FibEntry>> augmentation = Collections.emptyMap();

    public L2FibEntryBuilder() {
    }

    public L2FibEntryBuilder(L2FibEntry base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<L2FibEntry>>, Augmentation<L2FibEntry>> aug =((AugmentationHolder<L2FibEntry>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._physAddress = base.getPhysAddress();
        this._action = base.getAction();
        this._outgoingInterface = base.getOutgoingInterface();
        this._bridgedVirtualInterface = base.isBridgedVirtualInterface();
        this._staticConfig = base.isStaticConfig();
    }


    public L2FibEntryKey key() {
        return key;
    }
    
    public Class<? extends L2FibAction> getAction() {
        return _action;
    }
    
    public String getOutgoingInterface() {
        return _outgoingInterface;
    }
    
    public PhysAddress getPhysAddress() {
        return _physAddress;
    }
    
    public Boolean isBridgedVirtualInterface() {
        return _bridgedVirtualInterface;
    }
    
    public Boolean isStaticConfig() {
        return _staticConfig;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<L2FibEntry>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public L2FibEntryBuilder withKey(final L2FibEntryKey key) {
        this.key = key;
        return this;
    }
    
    public L2FibEntryBuilder setAction(final Class<? extends L2FibAction> value) {
        this._action = value;
        return this;
    }
    
    public L2FibEntryBuilder setOutgoingInterface(final String value) {
        this._outgoingInterface = value;
        return this;
    }
    
    public L2FibEntryBuilder setPhysAddress(final PhysAddress value) {
        this._physAddress = value;
        return this;
    }
    
    public L2FibEntryBuilder setBridgedVirtualInterface(final Boolean value) {
        this._bridgedVirtualInterface = value;
        return this;
    }
    
    public L2FibEntryBuilder setStaticConfig(final Boolean value) {
        this._staticConfig = value;
        return this;
    }
    
    public L2FibEntryBuilder addAugmentation(Class<? extends Augmentation<L2FibEntry>> augmentationType, Augmentation<L2FibEntry> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public L2FibEntryBuilder removeAugmentation(Class<? extends Augmentation<L2FibEntry>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public L2FibEntry build() {
        return new L2FibEntryImpl(this);
    }

    private static final class L2FibEntryImpl
        extends AbstractAugmentable<L2FibEntry>
        implements L2FibEntry {
    
        private final Class<? extends L2FibAction> _action;
        private final String _outgoingInterface;
        private final PhysAddress _physAddress;
        private final Boolean _bridgedVirtualInterface;
        private final Boolean _staticConfig;
        private final L2FibEntryKey key;
    
        L2FibEntryImpl(L2FibEntryBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new L2FibEntryKey(base.getPhysAddress());
            }
            this._physAddress = key.getPhysAddress();
            this._action = base.getAction();
            this._outgoingInterface = base.getOutgoingInterface();
            this._bridgedVirtualInterface = base.isBridgedVirtualInterface();
            this._staticConfig = base.isStaticConfig();
        }
    
        @Override
        public L2FibEntryKey key() {
            return key;
        }
        
        @Override
        public Class<? extends L2FibAction> getAction() {
            return _action;
        }
        
        @Override
        public String getOutgoingInterface() {
            return _outgoingInterface;
        }
        
        @Override
        public PhysAddress getPhysAddress() {
            return _physAddress;
        }
        
        @Override
        public Boolean isBridgedVirtualInterface() {
            return _bridgedVirtualInterface;
        }
        
        @Override
        public Boolean isStaticConfig() {
            return _staticConfig;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_action);
            result = prime * result + Objects.hashCode(_outgoingInterface);
            result = prime * result + Objects.hashCode(_physAddress);
            result = prime * result + Objects.hashCode(_bridgedVirtualInterface);
            result = prime * result + Objects.hashCode(_staticConfig);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!L2FibEntry.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            L2FibEntry other = (L2FibEntry)obj;
            if (!Objects.equals(_action, other.getAction())) {
                return false;
            }
            if (!Objects.equals(_outgoingInterface, other.getOutgoingInterface())) {
                return false;
            }
            if (!Objects.equals(_physAddress, other.getPhysAddress())) {
                return false;
            }
            if (!Objects.equals(_bridgedVirtualInterface, other.isBridgedVirtualInterface())) {
                return false;
            }
            if (!Objects.equals(_staticConfig, other.isStaticConfig())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                L2FibEntryImpl otherImpl = (L2FibEntryImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<L2FibEntry>>, Augmentation<L2FibEntry>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("L2FibEntry");
            CodeHelpers.appendValue(helper, "_action", _action);
            CodeHelpers.appendValue(helper, "_outgoingInterface", _outgoingInterface);
            CodeHelpers.appendValue(helper, "_physAddress", _physAddress);
            CodeHelpers.appendValue(helper, "_bridgedVirtualInterface", _bridgedVirtualInterface);
            CodeHelpers.appendValue(helper, "_staticConfig", _staticConfig);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
