package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.span.attributes.mirrored.interfaces;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.SpanState;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link MirroredInterfaceBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     MirroredInterfaceBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new MirroredInterfaceBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of MirroredInterfaceBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see MirroredInterfaceBuilder
 * @see Builder
 *
 */
public class MirroredInterfaceBuilder implements Builder<MirroredInterface> {

    private String _ifaceRef;
    private SpanState _state;
    private MirroredInterfaceKey key;


    Map<Class<? extends Augmentation<MirroredInterface>>, Augmentation<MirroredInterface>> augmentation = Collections.emptyMap();

    public MirroredInterfaceBuilder() {
    }

    public MirroredInterfaceBuilder(MirroredInterface base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<MirroredInterface>>, Augmentation<MirroredInterface>> aug =((AugmentationHolder<MirroredInterface>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._ifaceRef = base.getIfaceRef();
        this._state = base.getState();
    }


    public MirroredInterfaceKey key() {
        return key;
    }
    
    public String getIfaceRef() {
        return _ifaceRef;
    }
    
    public SpanState getState() {
        return _state;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<MirroredInterface>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public MirroredInterfaceBuilder withKey(final MirroredInterfaceKey key) {
        this.key = key;
        return this;
    }
    
    public MirroredInterfaceBuilder setIfaceRef(final String value) {
        this._ifaceRef = value;
        return this;
    }
    
    public MirroredInterfaceBuilder setState(final SpanState value) {
        this._state = value;
        return this;
    }
    
    public MirroredInterfaceBuilder addAugmentation(Class<? extends Augmentation<MirroredInterface>> augmentationType, Augmentation<MirroredInterface> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public MirroredInterfaceBuilder removeAugmentation(Class<? extends Augmentation<MirroredInterface>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public MirroredInterface build() {
        return new MirroredInterfaceImpl(this);
    }

    private static final class MirroredInterfaceImpl
        extends AbstractAugmentable<MirroredInterface>
        implements MirroredInterface {
    
        private final String _ifaceRef;
        private final SpanState _state;
        private final MirroredInterfaceKey key;
    
        MirroredInterfaceImpl(MirroredInterfaceBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new MirroredInterfaceKey(base.getIfaceRef());
            }
            this._ifaceRef = key.getIfaceRef();
            this._state = base.getState();
        }
    
        @Override
        public MirroredInterfaceKey key() {
            return key;
        }
        
        @Override
        public String getIfaceRef() {
            return _ifaceRef;
        }
        
        @Override
        public SpanState getState() {
            return _state;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_ifaceRef);
            result = prime * result + Objects.hashCode(_state);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!MirroredInterface.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            MirroredInterface other = (MirroredInterface)obj;
            if (!Objects.equals(_ifaceRef, other.getIfaceRef())) {
                return false;
            }
            if (!Objects.equals(_state, other.getState())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                MirroredInterfaceImpl otherImpl = (MirroredInterfaceImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<MirroredInterface>>, Augmentation<MirroredInterface>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("MirroredInterface");
            CodeHelpers.appendValue(helper, "_ifaceRef", _ifaceRef);
            CodeHelpers.appendValue(helper, "_state", _state);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
