package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.span.state.attributes;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.v3po.rev170607.span.state.attributes.mirrored.interfaces.MirroredInterface;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link MirroredInterfacesBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     MirroredInterfacesBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new MirroredInterfacesBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of MirroredInterfacesBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see MirroredInterfacesBuilder
 * @see Builder
 *
 */
public class MirroredInterfacesBuilder implements Builder<MirroredInterfaces> {

    private List<MirroredInterface> _mirroredInterface;


    Map<Class<? extends Augmentation<MirroredInterfaces>>, Augmentation<MirroredInterfaces>> augmentation = Collections.emptyMap();

    public MirroredInterfacesBuilder() {
    }

    public MirroredInterfacesBuilder(MirroredInterfaces base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<MirroredInterfaces>>, Augmentation<MirroredInterfaces>> aug =((AugmentationHolder<MirroredInterfaces>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._mirroredInterface = base.getMirroredInterface();
    }


    public List<MirroredInterface> getMirroredInterface() {
        return _mirroredInterface;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<MirroredInterfaces>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public MirroredInterfacesBuilder setMirroredInterface(final List<MirroredInterface> values) {
        this._mirroredInterface = values;
        return this;
    }
    
    
    public MirroredInterfacesBuilder addAugmentation(Class<? extends Augmentation<MirroredInterfaces>> augmentationType, Augmentation<MirroredInterfaces> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public MirroredInterfacesBuilder removeAugmentation(Class<? extends Augmentation<MirroredInterfaces>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public MirroredInterfaces build() {
        return new MirroredInterfacesImpl(this);
    }

    private static final class MirroredInterfacesImpl
        extends AbstractAugmentable<MirroredInterfaces>
        implements MirroredInterfaces {
    
        private final List<MirroredInterface> _mirroredInterface;
    
        MirroredInterfacesImpl(MirroredInterfacesBuilder base) {
            super(base.augmentation);
            this._mirroredInterface = base.getMirroredInterface();
        }
    
        @Override
        public List<MirroredInterface> getMirroredInterface() {
            return _mirroredInterface;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_mirroredInterface);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!MirroredInterfaces.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            MirroredInterfaces other = (MirroredInterfaces)obj;
            if (!Objects.equals(_mirroredInterface, other.getMirroredInterface())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                MirroredInterfacesImpl otherImpl = (MirroredInterfacesImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<MirroredInterfaces>>, Augmentation<MirroredInterfaces>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("MirroredInterfaces");
            CodeHelpers.appendValue(helper, "_mirroredInterface", _mirroredInterface);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
