package org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.vbridge.topology.rev160129;
import com.google.common.base.MoreObjects;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.opendaylight.params.xml.ns.yang.vbridge.topology.rev160129.network.topology.topology.node.BridgeMember;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link NodeVbridgeAugmentBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     NodeVbridgeAugmentBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new NodeVbridgeAugmentBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of NodeVbridgeAugmentBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see NodeVbridgeAugmentBuilder
 * @see Builder
 *
 */
public class NodeVbridgeAugmentBuilder implements Builder<NodeVbridgeAugment> {

    private BridgeMember _bridgeMember;



    public NodeVbridgeAugmentBuilder() {
    }

    public NodeVbridgeAugmentBuilder(NodeVbridgeAugment base) {
        this._bridgeMember = base.getBridgeMember();
    }


    public BridgeMember getBridgeMember() {
        return _bridgeMember;
    }

    
    public NodeVbridgeAugmentBuilder setBridgeMember(final BridgeMember value) {
        this._bridgeMember = value;
        return this;
    }
    

    @Override
    public NodeVbridgeAugment build() {
        return new NodeVbridgeAugmentImpl(this);
    }

    private static final class NodeVbridgeAugmentImpl
        implements NodeVbridgeAugment {
    
        private final BridgeMember _bridgeMember;
    
        NodeVbridgeAugmentImpl(NodeVbridgeAugmentBuilder base) {
            this._bridgeMember = base.getBridgeMember();
        }
    
        @Override
        public BridgeMember getBridgeMember() {
            return _bridgeMember;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_bridgeMember);
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!NodeVbridgeAugment.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            NodeVbridgeAugment other = (NodeVbridgeAugment)obj;
            if (!Objects.equals(_bridgeMember, other.getBridgeMember())) {
                return false;
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("NodeVbridgeAugment");
            CodeHelpers.appendValue(helper, "_bridgeMember", _bridgeMember);
            return helper.toString();
        }
    }
}
