/*
 * Copyright (c) 2014 Cisco Systems, Inc. and others.  All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 */

package org.opendaylight.netconf.confignetconfconnector.operations;

import com.google.common.base.Optional;
import org.opendaylight.controller.config.facade.xml.Datastore;
import org.opendaylight.controller.config.util.xml.DocumentedException;
import org.opendaylight.controller.config.util.xml.XmlElement;
import org.opendaylight.controller.config.util.xml.XmlUtil;
import org.opendaylight.netconf.api.xml.XmlNetconfConstants;
import org.opendaylight.netconf.util.mapping.AbstractLastNetconfOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Simple unlock implementation that pretends to unlock candidate datastore.
 * Candidate datastore is allocated per session and is private so no real locking is needed
 * (JMX is the only possible interference)
 */
public class UnLock extends AbstractLastNetconfOperation {

    private static final Logger LOG = LoggerFactory.getLogger(UnLock.class);

    private static final String UNLOCK = "unlock";

    public UnLock(final String netconfSessionIdForReporting) {
        super(netconfSessionIdForReporting);
    }

    @Override
    protected Element handleWithNoSubsequentOperations(final Document document,
                                                       final XmlElement operationElement) throws DocumentedException {
        final Datastore targetDatastore = Lock.extractTargetParameter(operationElement);
        if (targetDatastore == Datastore.candidate) {
            // Since candidate datastore instances are allocated per session and not accessible anywhere else,
            // no need to lock
            LOG.debug("Unlocking {} datastore on session: {}", targetDatastore, getNetconfSessionIdForReporting());
            // TODO this should fail if we are not locked
            return XmlUtil.createElement(document, XmlNetconfConstants.OK, Optional.absent());
        }

        // Not supported running lock
        throw new DocumentedException("Unable to unlock " + Datastore.running + " datastore",
                DocumentedException.ErrorType.APPLICATION,
                DocumentedException.ErrorTag.OPERATION_NOT_SUPPORTED, DocumentedException.ErrorSeverity.ERROR);
    }

    @Override
    protected String getOperationName() {
        return UNLOCK;
    }
}
