package org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.netmod.notification.rev080714.netconf.streams;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import java.lang.Boolean;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.netconf.notification._1._0.rev080714.StreamNameType;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.yang.types.rev130715.DateAndTime;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link StreamBuilder} instances.
 *
 * @see StreamBuilder
 *
 */
public class StreamBuilder implements Builder<Stream> {

    private String _description;
    private StreamNameType _name;
    private DateAndTime _replayLogCreationTime;
    private Boolean _replaySupport;
    private StreamKey key;


    Map<Class<? extends Augmentation<Stream>>, Augmentation<Stream>> augmentation = Collections.emptyMap();

    public StreamBuilder() {
    }

    public StreamBuilder(Stream base) {
        this.key = base.key();
        this._name = base.getName();
        this._description = base.getDescription();
        this._replayLogCreationTime = base.getReplayLogCreationTime();
        this._replaySupport = base.isReplaySupport();
        if (base instanceof StreamImpl) {
            StreamImpl impl = (StreamImpl) base;
            if (!impl.augmentation.isEmpty()) {
                this.augmentation = new HashMap<>(impl.augmentation);
            }
        } else if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Stream>>, Augmentation<Stream>> aug =((AugmentationHolder<Stream>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
    }


    public StreamKey key() {
        return key;
    }
    
    public String getDescription() {
        return _description;
    }
    
    public StreamNameType getName() {
        return _name;
    }
    
    public DateAndTime getReplayLogCreationTime() {
        return _replayLogCreationTime;
    }
    
    public Boolean isReplaySupport() {
        return _replaySupport;
    }
    
    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<Stream>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public StreamBuilder withKey(final StreamKey key) {
        this.key = key;
        return this;
    }
    
    public StreamBuilder setDescription(final String value) {
        this._description = value;
        return this;
    }
    
    public StreamBuilder setName(final StreamNameType value) {
        this._name = value;
        return this;
    }
    
    public StreamBuilder setReplayLogCreationTime(final DateAndTime value) {
        this._replayLogCreationTime = value;
        return this;
    }
    
    public StreamBuilder setReplaySupport(final Boolean value) {
        this._replaySupport = value;
        return this;
    }
    
    public StreamBuilder addAugmentation(Class<? extends Augmentation<Stream>> augmentationType, Augmentation<Stream> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public StreamBuilder removeAugmentation(Class<? extends Augmentation<Stream>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public Stream build() {
        return new StreamImpl(this);
    }

    private static final class StreamImpl implements Stream {
    
        private final String _description;
        private final StreamNameType _name;
        private final DateAndTime _replayLogCreationTime;
        private final Boolean _replaySupport;
        private final StreamKey key;
    
        private Map<Class<? extends Augmentation<Stream>>, Augmentation<Stream>> augmentation = Collections.emptyMap();
    
        StreamImpl(StreamBuilder base) {
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new StreamKey(base.getName());
            }
            this._name = key.getName();
            this._description = base.getDescription();
            this._replayLogCreationTime = base.getReplayLogCreationTime();
            this._replaySupport = base.isReplaySupport();
            this.augmentation = ImmutableMap.copyOf(base.augmentation);
        }
    
        @Override
        public Class<Stream> getImplementedInterface() {
            return Stream.class;
        }
    
        @Override
        public StreamKey key() {
            return key;
        }
        
        @Override
        public String getDescription() {
            return _description;
        }
        
        @Override
        public StreamNameType getName() {
            return _name;
        }
        
        @Override
        public DateAndTime getReplayLogCreationTime() {
            return _replayLogCreationTime;
        }
        
        @Override
        public Boolean isReplaySupport() {
            return _replaySupport;
        }
        
        @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
        @Override
        public <E$$ extends Augmentation<Stream>> E$$ augmentation(Class<E$$> augmentationType) {
            return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_description);
            result = prime * result + Objects.hashCode(_name);
            result = prime * result + Objects.hashCode(_replayLogCreationTime);
            result = prime * result + Objects.hashCode(_replaySupport);
            result = prime * result + Objects.hashCode(augmentation);
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Stream.class.equals(((DataObject)obj).getImplementedInterface())) {
                return false;
            }
            Stream other = (Stream)obj;
            if (!Objects.equals(_description, other.getDescription())) {
                return false;
            }
            if (!Objects.equals(_name, other.getName())) {
                return false;
            }
            if (!Objects.equals(_replayLogCreationTime, other.getReplayLogCreationTime())) {
                return false;
            }
            if (!Objects.equals(_replaySupport, other.isReplaySupport())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                StreamImpl otherImpl = (StreamImpl) obj;
                if (!Objects.equals(augmentation, otherImpl.augmentation)) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Stream>>, Augmentation<Stream>> e : augmentation.entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Stream");
            CodeHelpers.appendValue(helper, "_description", _description);
            CodeHelpers.appendValue(helper, "_name", _name);
            CodeHelpers.appendValue(helper, "_replayLogCreationTime", _replayLogCreationTime);
            CodeHelpers.appendValue(helper, "_replaySupport", _replaySupport);
            CodeHelpers.appendValue(helper, "augmentation", augmentation.values());
            return helper.toString();
        }
    }
}
