package org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.netconf.notification._1._0.rev080714;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.netconf.notification._1._0.rev080714.create.subscription.input.Filter;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.yang.types.rev130715.DateAndTime;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link CreateSubscriptionInputBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     CreateSubscriptionInputBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new CreateSubscriptionInputBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of CreateSubscriptionInputBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see CreateSubscriptionInputBuilder
 * @see Builder
 *
 */
public class CreateSubscriptionInputBuilder implements Builder<CreateSubscriptionInput> {

    private Filter _filter;
    private DateAndTime _startTime;
    private DateAndTime _stopTime;
    private StreamNameType _stream;


    Map<Class<? extends Augmentation<CreateSubscriptionInput>>, Augmentation<CreateSubscriptionInput>> augmentation = Collections.emptyMap();

    public CreateSubscriptionInputBuilder() {
    }

    public CreateSubscriptionInputBuilder(CreateSubscriptionInput base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<CreateSubscriptionInput>>, Augmentation<CreateSubscriptionInput>> aug =((AugmentationHolder<CreateSubscriptionInput>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._filter = base.getFilter();
        this._startTime = base.getStartTime();
        this._stopTime = base.getStopTime();
        this._stream = base.getStream();
    }


    public Filter getFilter() {
        return _filter;
    }
    
    public DateAndTime getStartTime() {
        return _startTime;
    }
    
    public DateAndTime getStopTime() {
        return _stopTime;
    }
    
    public StreamNameType getStream() {
        return _stream;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<CreateSubscriptionInput>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public CreateSubscriptionInputBuilder setFilter(final Filter value) {
        this._filter = value;
        return this;
    }
    
    public CreateSubscriptionInputBuilder setStartTime(final DateAndTime value) {
        this._startTime = value;
        return this;
    }
    
    public CreateSubscriptionInputBuilder setStopTime(final DateAndTime value) {
        this._stopTime = value;
        return this;
    }
    
    public CreateSubscriptionInputBuilder setStream(final StreamNameType value) {
        this._stream = value;
        return this;
    }
    
    public CreateSubscriptionInputBuilder addAugmentation(Class<? extends Augmentation<CreateSubscriptionInput>> augmentationType, Augmentation<CreateSubscriptionInput> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public CreateSubscriptionInputBuilder removeAugmentation(Class<? extends Augmentation<CreateSubscriptionInput>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public CreateSubscriptionInput build() {
        return new CreateSubscriptionInputImpl(this);
    }

    private static final class CreateSubscriptionInputImpl
        extends AbstractAugmentable<CreateSubscriptionInput>
        implements CreateSubscriptionInput {
    
        private final Filter _filter;
        private final DateAndTime _startTime;
        private final DateAndTime _stopTime;
        private final StreamNameType _stream;
    
        CreateSubscriptionInputImpl(CreateSubscriptionInputBuilder base) {
            super(base.augmentation);
            this._filter = base.getFilter();
            this._startTime = base.getStartTime();
            this._stopTime = base.getStopTime();
            this._stream = base.getStream();
        }
    
        @Override
        public Filter getFilter() {
            return _filter;
        }
        
        @Override
        public DateAndTime getStartTime() {
            return _startTime;
        }
        
        @Override
        public DateAndTime getStopTime() {
            return _stopTime;
        }
        
        @Override
        public StreamNameType getStream() {
            return _stream;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_filter);
            result = prime * result + Objects.hashCode(_startTime);
            result = prime * result + Objects.hashCode(_stopTime);
            result = prime * result + Objects.hashCode(_stream);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!CreateSubscriptionInput.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            CreateSubscriptionInput other = (CreateSubscriptionInput)obj;
            if (!Objects.equals(_filter, other.getFilter())) {
                return false;
            }
            if (!Objects.equals(_startTime, other.getStartTime())) {
                return false;
            }
            if (!Objects.equals(_stopTime, other.getStopTime())) {
                return false;
            }
            if (!Objects.equals(_stream, other.getStream())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                CreateSubscriptionInputImpl otherImpl = (CreateSubscriptionInputImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<CreateSubscriptionInput>>, Augmentation<CreateSubscriptionInput>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("CreateSubscriptionInput");
            CodeHelpers.appendValue(helper, "_filter", _filter);
            CodeHelpers.appendValue(helper, "_startTime", _startTime);
            CodeHelpers.appendValue(helper, "_stopTime", _stopTime);
            CodeHelpers.appendValue(helper, "_stream", _stream);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
