package org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.netmod.notification.rev080714.netconf.streams;
import com.google.common.base.MoreObjects;
import java.lang.Boolean;
import java.lang.Class;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.netconf.notification._1._0.rev080714.StreamNameType;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.yang.types.rev130715.DateAndTime;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link StreamBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     StreamBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new StreamBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of StreamBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see StreamBuilder
 * @see Builder
 *
 */
public class StreamBuilder implements Builder<Stream> {

    private String _description;
    private StreamNameType _name;
    private DateAndTime _replayLogCreationTime;
    private Boolean _replaySupport;
    private StreamKey key;


    Map<Class<? extends Augmentation<Stream>>, Augmentation<Stream>> augmentation = Collections.emptyMap();

    public StreamBuilder() {
    }

    public StreamBuilder(Stream base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<Stream>>, Augmentation<Stream>> aug =((AugmentationHolder<Stream>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this.key = base.key();
        this._name = base.getName();
        this._description = base.getDescription();
        this._replayLogCreationTime = base.getReplayLogCreationTime();
        this._replaySupport = base.isReplaySupport();
    }


    public StreamKey key() {
        return key;
    }
    
    public String getDescription() {
        return _description;
    }
    
    public StreamNameType getName() {
        return _name;
    }
    
    public DateAndTime getReplayLogCreationTime() {
        return _replayLogCreationTime;
    }
    
    public Boolean isReplaySupport() {
        return _replaySupport;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<Stream>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    public StreamBuilder withKey(final StreamKey key) {
        this.key = key;
        return this;
    }
    
    public StreamBuilder setDescription(final String value) {
        this._description = value;
        return this;
    }
    
    public StreamBuilder setName(final StreamNameType value) {
        this._name = value;
        return this;
    }
    
    public StreamBuilder setReplayLogCreationTime(final DateAndTime value) {
        this._replayLogCreationTime = value;
        return this;
    }
    
    public StreamBuilder setReplaySupport(final Boolean value) {
        this._replaySupport = value;
        return this;
    }
    
    public StreamBuilder addAugmentation(Class<? extends Augmentation<Stream>> augmentationType, Augmentation<Stream> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public StreamBuilder removeAugmentation(Class<? extends Augmentation<Stream>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public Stream build() {
        return new StreamImpl(this);
    }

    private static final class StreamImpl
        extends AbstractAugmentable<Stream>
        implements Stream {
    
        private final String _description;
        private final StreamNameType _name;
        private final DateAndTime _replayLogCreationTime;
        private final Boolean _replaySupport;
        private final StreamKey key;
    
        StreamImpl(StreamBuilder base) {
            super(base.augmentation);
            if (base.key() != null) {
                this.key = base.key();
            } else {
                this.key = new StreamKey(base.getName());
            }
            this._name = key.getName();
            this._description = base.getDescription();
            this._replayLogCreationTime = base.getReplayLogCreationTime();
            this._replaySupport = base.isReplaySupport();
        }
    
        @Override
        public StreamKey key() {
            return key;
        }
        
        @Override
        public String getDescription() {
            return _description;
        }
        
        @Override
        public StreamNameType getName() {
            return _name;
        }
        
        @Override
        public DateAndTime getReplayLogCreationTime() {
            return _replayLogCreationTime;
        }
        
        @Override
        public Boolean isReplaySupport() {
            return _replaySupport;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_description);
            result = prime * result + Objects.hashCode(_name);
            result = prime * result + Objects.hashCode(_replayLogCreationTime);
            result = prime * result + Objects.hashCode(_replaySupport);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!Stream.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            Stream other = (Stream)obj;
            if (!Objects.equals(_description, other.getDescription())) {
                return false;
            }
            if (!Objects.equals(_name, other.getName())) {
                return false;
            }
            if (!Objects.equals(_replayLogCreationTime, other.getReplayLogCreationTime())) {
                return false;
            }
            if (!Objects.equals(_replaySupport, other.isReplaySupport())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                StreamImpl otherImpl = (StreamImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<Stream>>, Augmentation<Stream>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("Stream");
            CodeHelpers.appendValue(helper, "_description", _description);
            CodeHelpers.appendValue(helper, "_name", _name);
            CodeHelpers.appendValue(helper, "_replayLogCreationTime", _replayLogCreationTime);
            CodeHelpers.appendValue(helper, "_replaySupport", _replaySupport);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
