package org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.netconf.notifications.rev120206;
import com.google.common.base.MoreObjects;
import java.lang.Class;
import java.lang.Long;
import java.lang.Object;
import java.lang.Override;
import java.lang.String;
import java.lang.SuppressWarnings;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.netconf.base._1._0.rev110601.SessionIdOrZeroType;
import org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.inet.types.rev130715.IpAddress;
import org.opendaylight.yangtools.concepts.Builder;
import org.opendaylight.yangtools.yang.binding.AbstractAugmentable;
import org.opendaylight.yangtools.yang.binding.Augmentation;
import org.opendaylight.yangtools.yang.binding.AugmentationHolder;
import org.opendaylight.yangtools.yang.binding.CodeHelpers;
import org.opendaylight.yangtools.yang.binding.DataObject;

/**
 * Class that builds {@link NetconfConfirmedCommitBuilder} instances. Overall design of the class is that of a
 * <a href="https://en.wikipedia.org/wiki/Fluent_interface">fluent interface</a>, where method chaining is used.
 *
 * <p>
 * In general, this class is supposed to be used like this template:
 * <pre>
 *   <code>
 *     NetconfConfirmedCommitBuilder createTarget(int fooXyzzy, int barBaz) {
 *         return new NetconfConfirmedCommitBuilderBuilder()
 *             .setFoo(new FooBuilder().setXyzzy(fooXyzzy).build())
 *             .setBar(new BarBuilder().setBaz(barBaz).build())
 *             .build();
 *     }
 *   </code>
 * </pre>
 *
 * <p>
 * This pattern is supported by the immutable nature of NetconfConfirmedCommitBuilder, as instances can be freely passed around without
 * worrying about synchronization issues.
 *
 * <p>
 * As a side note: method chaining results in:
 * <ul>
 *   <li>very efficient Java bytecode, as the method invocation result, in this case the Builder reference, is
 *       on the stack, so further method invocations just need to fill method arguments for the next method
 *       invocation, which is terminated by {@link #build()}, which is then returned from the method</li>
 *   <li>better understanding by humans, as the scope of mutable state (the builder) is kept to a minimum and is
 *       very localized</li>
 *   <li>better optimization oportunities, as the object scope is minimized in terms of invocation (rather than
 *       method) stack, making <a href="https://en.wikipedia.org/wiki/Escape_analysis">escape analysis</a> a lot
 *       easier. Given enough compiler (JIT/AOT) prowess, the cost of th builder object can be completely
 *       eliminated</li>
 * </ul>
 *
 * @see NetconfConfirmedCommitBuilder
 * @see Builder
 *
 */
public class NetconfConfirmedCommitBuilder implements Builder<NetconfConfirmedCommit> {

    private NetconfConfirmedCommit.ConfirmEvent _confirmEvent;
    private SessionIdOrZeroType _sessionId;
    private IpAddress _sourceHost;
    private Long _timeout;
    private String _username;


    Map<Class<? extends Augmentation<NetconfConfirmedCommit>>, Augmentation<NetconfConfirmedCommit>> augmentation = Collections.emptyMap();

    public NetconfConfirmedCommitBuilder() {
    }
    public NetconfConfirmedCommitBuilder(org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.netconf.notifications.rev120206.CommonSessionParms arg) {
        this._username = arg.getUsername();
        this._sessionId = arg.getSessionId();
        this._sourceHost = arg.getSourceHost();
    }

    public NetconfConfirmedCommitBuilder(NetconfConfirmedCommit base) {
        if (base instanceof AugmentationHolder) {
            @SuppressWarnings("unchecked")
            Map<Class<? extends Augmentation<NetconfConfirmedCommit>>, Augmentation<NetconfConfirmedCommit>> aug =((AugmentationHolder<NetconfConfirmedCommit>) base).augmentations();
            if (!aug.isEmpty()) {
                this.augmentation = new HashMap<>(aug);
            }
        }
        this._confirmEvent = base.getConfirmEvent();
        this._sessionId = base.getSessionId();
        this._sourceHost = base.getSourceHost();
        this._timeout = base.getTimeout();
        this._username = base.getUsername();
    }

    /**
     * Set fields from given grouping argument. Valid argument is instance of one of following types:
     * <ul>
     * <li>org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.netconf.notifications.rev120206.CommonSessionParms</li>
     * </ul>
     *
     * @param arg grouping object
     * @throws IllegalArgumentException if given argument is none of valid types
    */
    public void fieldsFrom(DataObject arg) {
        boolean isValidArg = false;
        if (arg instanceof org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.netconf.notifications.rev120206.CommonSessionParms) {
            this._username = ((org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.netconf.notifications.rev120206.CommonSessionParms)arg).getUsername();
            this._sessionId = ((org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.netconf.notifications.rev120206.CommonSessionParms)arg).getSessionId();
            this._sourceHost = ((org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.netconf.notifications.rev120206.CommonSessionParms)arg).getSourceHost();
            isValidArg = true;
        }
        CodeHelpers.validValue(isValidArg, arg, "[org.opendaylight.yang.gen.v1.urn.ietf.params.xml.ns.yang.ietf.netconf.notifications.rev120206.CommonSessionParms]");
    }

    public NetconfConfirmedCommit.ConfirmEvent getConfirmEvent() {
        return _confirmEvent;
    }
    
    public SessionIdOrZeroType getSessionId() {
        return _sessionId;
    }
    
    public IpAddress getSourceHost() {
        return _sourceHost;
    }
    
    public Long getTimeout() {
        return _timeout;
    }
    
    public String getUsername() {
        return _username;
    }

    @SuppressWarnings({ "unchecked", "checkstyle:methodTypeParameterName"})
    public <E$$ extends Augmentation<NetconfConfirmedCommit>> E$$ augmentation(Class<E$$> augmentationType) {
        return (E$$) augmentation.get(CodeHelpers.nonNullValue(augmentationType, "augmentationType"));
    }

    
    public NetconfConfirmedCommitBuilder setConfirmEvent(final NetconfConfirmedCommit.ConfirmEvent value) {
        this._confirmEvent = value;
        return this;
    }
    
    public NetconfConfirmedCommitBuilder setSessionId(final SessionIdOrZeroType value) {
        this._sessionId = value;
        return this;
    }
    
    public NetconfConfirmedCommitBuilder setSourceHost(final IpAddress value) {
        this._sourceHost = value;
        return this;
    }
    private static void checkTimeoutRange(final long value) {
        if (value >= 0L && value <= 4294967295L) {
            return;
        }
        CodeHelpers.throwInvalidRange("[[0..4294967295]]", value);
    }
    
    public NetconfConfirmedCommitBuilder setTimeout(final Long value) {
    if (value != null) {
        checkTimeoutRange(value);
        
    }
        this._timeout = value;
        return this;
    }
    
    public NetconfConfirmedCommitBuilder setUsername(final String value) {
        this._username = value;
        return this;
    }
    
    public NetconfConfirmedCommitBuilder addAugmentation(Class<? extends Augmentation<NetconfConfirmedCommit>> augmentationType, Augmentation<NetconfConfirmedCommit> augmentationValue) {
        if (augmentationValue == null) {
            return removeAugmentation(augmentationType);
        }
    
        if (!(this.augmentation instanceof HashMap)) {
            this.augmentation = new HashMap<>();
        }
    
        this.augmentation.put(augmentationType, augmentationValue);
        return this;
    }
    
    public NetconfConfirmedCommitBuilder removeAugmentation(Class<? extends Augmentation<NetconfConfirmedCommit>> augmentationType) {
        if (this.augmentation instanceof HashMap) {
            this.augmentation.remove(augmentationType);
        }
        return this;
    }

    @Override
    public NetconfConfirmedCommit build() {
        return new NetconfConfirmedCommitImpl(this);
    }

    private static final class NetconfConfirmedCommitImpl
        extends AbstractAugmentable<NetconfConfirmedCommit>
        implements NetconfConfirmedCommit {
    
        private final NetconfConfirmedCommit.ConfirmEvent _confirmEvent;
        private final SessionIdOrZeroType _sessionId;
        private final IpAddress _sourceHost;
        private final Long _timeout;
        private final String _username;
    
        NetconfConfirmedCommitImpl(NetconfConfirmedCommitBuilder base) {
            super(base.augmentation);
            this._confirmEvent = base.getConfirmEvent();
            this._sessionId = base.getSessionId();
            this._sourceHost = base.getSourceHost();
            this._timeout = base.getTimeout();
            this._username = base.getUsername();
        }
    
        @Override
        public NetconfConfirmedCommit.ConfirmEvent getConfirmEvent() {
            return _confirmEvent;
        }
        
        @Override
        public SessionIdOrZeroType getSessionId() {
            return _sessionId;
        }
        
        @Override
        public IpAddress getSourceHost() {
            return _sourceHost;
        }
        
        @Override
        public Long getTimeout() {
            return _timeout;
        }
        
        @Override
        public String getUsername() {
            return _username;
        }
    
        private int hash = 0;
        private volatile boolean hashValid = false;
        
        @Override
        public int hashCode() {
            if (hashValid) {
                return hash;
            }
        
            final int prime = 31;
            int result = 1;
            result = prime * result + Objects.hashCode(_confirmEvent);
            result = prime * result + Objects.hashCode(_sessionId);
            result = prime * result + Objects.hashCode(_sourceHost);
            result = prime * result + Objects.hashCode(_timeout);
            result = prime * result + Objects.hashCode(_username);
            result = prime * result + Objects.hashCode(augmentations());
        
            hash = result;
            hashValid = true;
            return result;
        }
    
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof DataObject)) {
                return false;
            }
            if (!NetconfConfirmedCommit.class.equals(((DataObject)obj).implementedInterface())) {
                return false;
            }
            NetconfConfirmedCommit other = (NetconfConfirmedCommit)obj;
            if (!Objects.equals(_confirmEvent, other.getConfirmEvent())) {
                return false;
            }
            if (!Objects.equals(_sessionId, other.getSessionId())) {
                return false;
            }
            if (!Objects.equals(_sourceHost, other.getSourceHost())) {
                return false;
            }
            if (!Objects.equals(_timeout, other.getTimeout())) {
                return false;
            }
            if (!Objects.equals(_username, other.getUsername())) {
                return false;
            }
            if (getClass() == obj.getClass()) {
                // Simple case: we are comparing against self
                NetconfConfirmedCommitImpl otherImpl = (NetconfConfirmedCommitImpl) obj;
                if (!Objects.equals(augmentations(), otherImpl.augmentations())) {
                    return false;
                }
            } else {
                // Hard case: compare our augments with presence there...
                for (Map.Entry<Class<? extends Augmentation<NetconfConfirmedCommit>>, Augmentation<NetconfConfirmedCommit>> e : augmentations().entrySet()) {
                    if (!e.getValue().equals(other.augmentation(e.getKey()))) {
                        return false;
                    }
                }
                // .. and give the other one the chance to do the same
                if (!obj.equals(this)) {
                    return false;
                }
            }
            return true;
        }
    
        @Override
        public String toString() {
            final MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper("NetconfConfirmedCommit");
            CodeHelpers.appendValue(helper, "_confirmEvent", _confirmEvent);
            CodeHelpers.appendValue(helper, "_sessionId", _sessionId);
            CodeHelpers.appendValue(helper, "_sourceHost", _sourceHost);
            CodeHelpers.appendValue(helper, "_timeout", _timeout);
            CodeHelpers.appendValue(helper, "_username", _username);
            CodeHelpers.appendValue(helper, "augmentation", augmentations().values());
            return helper.toString();
        }
    }
}
