/*
 * Copyright (c) 2017 Brocade Communication Systems and others.  All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 */
package org.opendaylight.netconf.nettyutil.handler.ssh.authentication;

import com.google.common.base.Strings;
import java.io.IOException;
import java.security.KeyPair;
import org.opendaylight.aaa.encrypt.PKIUtil;
import org.opendaylight.netconf.shaded.sshd.client.future.AuthFuture;
import org.opendaylight.netconf.shaded.sshd.client.session.ClientSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Represents Auth information for the public key based authentication for netconf.
 */
public class PublicKeyAuth extends LoginPasswordHandler {
    private KeyPair keyPair = null;
    private static final Logger LOG = LoggerFactory.getLogger(PublicKeyAuth.class);

    public PublicKeyAuth(final String username, final String password, final String keyPath,
                         String passPhrase) {
        super(username, password);
        try {
            boolean isKeyPathAbsent = Strings.isNullOrEmpty(keyPath);
            passPhrase = Strings.isNullOrEmpty(passPhrase) ? "" : passPhrase;
            if (!isKeyPathAbsent) {
                this.keyPair = new PKIUtil().decodePrivateKey(keyPath, passPhrase);
            } else {
                LOG.info("Private key path not specified in the config file.");
            }
        } catch (IOException ioEx) {
            LOG.warn("Not able to read the private key and passphrase for netconf client", ioEx);
        }
    }

    @Override
    public AuthFuture authenticate(final ClientSession session) throws IOException {
        if (keyPair != null) {
            session.addPublicKeyIdentity(keyPair);
        }

        return super.authenticate(session);
    }
}
