/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.netconf.nettyutil;

import com.google.common.annotations.Beta;
import com.google.common.base.Preconditions;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.handler.ssl.SslHandler;
import io.netty.util.Timeout;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.Promise;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import org.checkerframework.checker.index.qual.NonNegative;
import org.checkerframework.checker.lock.qual.GuardedBy;
import org.checkerframework.checker.lock.qual.Holding;
import org.eclipse.jdt.annotation.NonNull;
import org.eclipse.jdt.annotation.Nullable;
import org.opendaylight.netconf.api.NetconfDocumentedException;
import org.opendaylight.netconf.api.NetconfSessionListener;
import org.opendaylight.netconf.api.messages.HelloMessage;
import org.opendaylight.netconf.api.messages.NetconfMessage;
import org.opendaylight.netconf.common.NetconfTimer;
import org.opendaylight.netconf.nettyutil.AbstractNetconfSession;
import org.opendaylight.netconf.nettyutil.handler.ChunkedFramingMechanismEncoder;
import org.opendaylight.netconf.nettyutil.handler.NetconfChunkAggregator;
import org.opendaylight.netconf.nettyutil.handler.NetconfMessageToXMLEncoder;
import org.opendaylight.netconf.nettyutil.handler.NetconfXMLToHelloMessageDecoder;
import org.opendaylight.netconf.nettyutil.handler.NetconfXMLToMessageDecoder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

public abstract class NetconfSessionNegotiator<S extends AbstractNetconfSession<S, L>, L extends NetconfSessionListener<S>>
extends ChannelInboundHandlerAdapter {
    private static final Logger LOG = LoggerFactory.getLogger(NetconfSessionNegotiator.class);
    private static final String NAME_OF_EXCEPTION_HANDLER = "lastExceptionHandler";
    private static final String DEFAULT_MAXIMUM_CHUNK_SIZE_PROP = "org.opendaylight.netconf.default.maximum.chunk.size";
    private static final int DEFAULT_MAXIMUM_CHUNK_SIZE_DEFAULT = 0x1000000;
    @Beta
    public static final @NonNegative int DEFAULT_MAXIMUM_INCOMING_CHUNK_SIZE;
    private final @NonNull HelloMessage localHello;
    protected final Channel channel;
    private final @NonNegative int maximumIncomingChunkSize;
    private final long connectionTimeoutMillis;
    private final Promise<S> promise;
    private final L sessionListener;
    private final NetconfTimer timer;
    private @GuardedBy(value={"this"}) Timeout timeoutTask;
    private @GuardedBy(value={"this"}) State state = State.IDLE;

    protected NetconfSessionNegotiator(HelloMessage hello, Promise<S> promise, Channel channel, NetconfTimer timer, L sessionListener, long connectionTimeoutMillis, @NonNegative int maximumIncomingChunkSize) {
        this.localHello = Objects.requireNonNull(hello);
        this.promise = Objects.requireNonNull(promise);
        this.channel = Objects.requireNonNull(channel);
        this.timer = Objects.requireNonNull(timer);
        this.sessionListener = sessionListener;
        this.connectionTimeoutMillis = connectionTimeoutMillis;
        this.maximumIncomingChunkSize = maximumIncomingChunkSize;
        Preconditions.checkArgument((maximumIncomingChunkSize > 0 ? 1 : 0) != 0, (String)"Invalid maximum incoming chunk size %s", (int)maximumIncomingChunkSize);
    }

    protected final @NonNull HelloMessage localHello() {
        return this.localHello;
    }

    protected final void startNegotiation() {
        if (this.ifNegotiatedAlready()) {
            LOG.debug("Negotiation on channel {} already started", (Object)this.channel);
        } else {
            SslHandler sslHandler = NetconfSessionNegotiator.getSslHandler(this.channel);
            if (sslHandler != null) {
                sslHandler.handshakeFuture().addListener(future -> {
                    Preconditions.checkState((boolean)future.isSuccess(), (Object)"Ssl handshake was not successful");
                    LOG.debug("Ssl handshake complete");
                    this.start();
                });
            } else {
                this.start();
            }
        }
    }

    protected final boolean ifNegotiatedAlready() {
        return this.state() != State.IDLE;
    }

    private synchronized State state() {
        return this.state;
    }

    private static @Nullable SslHandler getSslHandler(Channel channel) {
        return (SslHandler)channel.pipeline().get(SslHandler.class);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void start() {
        LOG.debug("Sending negotiation proposal {} on channel {}", (Object)this.localHello, (Object)this.channel);
        ChannelFuture helloFuture = this.channel.writeAndFlush((Object)this.localHello);
        Throwable helloCause = helloFuture.cause();
        if (helloCause != null) {
            LOG.warn("Failed to send negotiation proposal on channel {}", (Object)this.channel, (Object)helloCause);
            this.failAndClose();
            return;
        }
        final class ExceptionHandlingInboundChannelHandler
        extends ChannelInboundHandlerAdapter {
            ExceptionHandlingInboundChannelHandler() {
            }

            public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
                LOG.warn("An exception occurred during negotiation with {} on channel {}", new Object[]{NetconfSessionNegotiator.this.channel.remoteAddress(), NetconfSessionNegotiator.this.channel, cause});
                NetconfSessionNegotiator.this.cancelTimeout();
                NetconfSessionNegotiator.this.negotiationFailed(cause);
                NetconfSessionNegotiator.this.changeState(State.FAILED);
            }
        }
        this.channel.pipeline().addLast(NAME_OF_EXCEPTION_HANDLER, (ChannelHandler)new ExceptionHandlingInboundChannelHandler());
        NetconfSessionNegotiator.replaceChannelHandler(this.channel, "netconfMessageEncoder", (ChannelHandler)new NetconfMessageToXMLEncoder());
        NetconfSessionNegotiator netconfSessionNegotiator = this;
        synchronized (netconfSessionNegotiator) {
            this.lockedChangeState(State.OPEN_WAIT);
            this.timeoutTask = this.timer.newTimeout(unused -> this.channel.eventLoop().execute(this::timeoutExpired), this.connectionTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        LOG.debug("Session negotiation started on channel {}", (Object)this.channel);
        helloFuture.addListener(this::onHelloWriteComplete);
    }

    private void onHelloWriteComplete(Future<?> future) {
        Throwable cause = future.cause();
        if (cause != null) {
            LOG.info("Failed to send message {} on channel {}", new Object[]{this.localHello, this.channel, cause});
            this.negotiationFailed(cause);
        } else {
            LOG.trace("Message {} sent to socket on channel {}", (Object)this.localHello, (Object)this.channel);
        }
    }

    private synchronized void timeoutExpired() {
        if (this.timeoutTask == null) {
            return;
        }
        this.timeoutTask = null;
        if (this.state != State.ESTABLISHED) {
            LOG.debug("Connection timeout after {}ms, session backed by channel {} is in state {}", new Object[]{this.connectionTimeoutMillis, this.channel, this.state});
            if (!this.promise.isDone() && !this.promise.isCancelled()) {
                LOG.warn("Netconf session backed by channel {} was not established after {}", (Object)this.channel, (Object)this.connectionTimeoutMillis);
                this.failAndClose();
            }
        } else if (this.channel.isOpen()) {
            this.channel.pipeline().remove(NAME_OF_EXCEPTION_HANDLER);
        }
    }

    private void failAndClose() {
        this.changeState(State.FAILED);
        this.channel.close().addListener(this::onChannelClosed);
    }

    private void onChannelClosed(Future<?> future) {
        Throwable cause = future.cause();
        if (cause != null) {
            LOG.warn("Channel {} closed: fail", (Object)this.channel, (Object)cause);
        } else {
            LOG.debug("Channel {} closed: success", (Object)this.channel);
        }
    }

    private synchronized void cancelTimeout() {
        if (this.timeoutTask != null && !this.timeoutTask.cancel()) {
            this.timeoutTask = null;
        }
    }

    protected final S getSessionForHelloMessage(HelloMessage netconfMessage) throws NetconfDocumentedException {
        Document doc = netconfMessage.getDocument();
        if (this.shouldUseChunkFraming(doc)) {
            this.insertChunkFramingToPipeline();
        }
        this.changeState(State.ESTABLISHED);
        return this.getSession(this.sessionListener, this.channel, netconfMessage);
    }

    protected abstract S getSession(L var1, Channel var2, HelloMessage var3) throws NetconfDocumentedException;

    private void insertChunkFramingToPipeline() {
        NetconfSessionNegotiator.replaceChannelHandler(this.channel, "frameEncoder", (ChannelHandler)new ChunkedFramingMechanismEncoder());
        NetconfSessionNegotiator.replaceChannelHandler(this.channel, "aggregator", (ChannelHandler)new NetconfChunkAggregator(this.maximumIncomingChunkSize));
    }

    private boolean shouldUseChunkFraming(Document doc) {
        return NetconfSessionNegotiator.containsBase11Capability(doc) && NetconfSessionNegotiator.containsBase11Capability(this.localHello.getDocument());
    }

    protected final void replaceHelloMessageInboundHandler(S session) {
        ChannelHandler helloMessageHandler = NetconfSessionNegotiator.replaceChannelHandler(this.channel, "netconfMessageDecoder", (ChannelHandler)new NetconfXMLToMessageDecoder());
        Preconditions.checkState((boolean)(helloMessageHandler instanceof NetconfXMLToHelloMessageDecoder), (String)"Pipeline handlers misplaced on session: %s, pipeline: %s", session, (Object)this.channel.pipeline());
        Iterable<NetconfMessage> netconfMessagesFromNegotiation = ((NetconfXMLToHelloMessageDecoder)helloMessageHandler).getPostHelloNetconfMessages();
        for (NetconfMessage message : netconfMessagesFromNegotiation) {
            ((AbstractNetconfSession)((Object)session)).handleMessage(message);
        }
    }

    private static ChannelHandler replaceChannelHandler(Channel channel, String handlerKey, ChannelHandler decoder) {
        return channel.pipeline().replace(handlerKey, handlerKey, decoder);
    }

    private synchronized void changeState(State newState) {
        this.lockedChangeState(newState);
    }

    @Holding(value={"this"})
    private void lockedChangeState(State newState) {
        LOG.debug("Changing state from : {} to : {} for channel: {}", new Object[]{this.state, newState, this.channel});
        Preconditions.checkState((boolean)NetconfSessionNegotiator.isStateChangePermitted(this.state, newState), (String)"Cannot change state from %s to %s for channel %s", (Object)((Object)this.state), (Object)((Object)newState), (Object)this.channel);
        this.state = newState;
    }

    private static boolean containsBase11Capability(Document doc) {
        NodeList nList = doc.getElementsByTagNameNS("urn:ietf:params:xml:ns:netconf:base:1.0", "capability");
        for (int i = 0; i < nList.getLength(); ++i) {
            if (!nList.item(i).getTextContent().contains("urn:ietf:params:netconf:base:1.1")) continue;
            return true;
        }
        return false;
    }

    private static boolean isStateChangePermitted(State state, State newState) {
        if (state == State.IDLE && (newState == State.OPEN_WAIT || newState == State.FAILED)) {
            return true;
        }
        if (state == State.OPEN_WAIT && (newState == State.ESTABLISHED || newState == State.FAILED)) {
            return true;
        }
        LOG.debug("Transition from {} to {} is not allowed", (Object)state, (Object)newState);
        return false;
    }

    protected final void negotiationSuccessful(S session) {
        LOG.debug("Negotiation on channel {} successful with session {}", (Object)this.channel, session);
        this.channel.pipeline().replace((ChannelHandler)this, "session", session);
        this.promise.setSuccess(session);
    }

    protected void negotiationFailed(Throwable cause) {
        LOG.debug("Negotiation on channel {} failed", (Object)this.channel, (Object)cause);
        this.channel.close();
        this.promise.setFailure(cause);
    }

    public final void channelActive(ChannelHandlerContext ctx) {
        LOG.debug("Starting session negotiation on channel {}", (Object)this.channel);
        try {
            this.startNegotiation();
        }
        catch (Exception e) {
            LOG.warn("Unexpected negotiation failure on channel {}", (Object)this.channel, (Object)e);
            this.negotiationFailed(e);
        }
    }

    public final void channelRead(ChannelHandlerContext ctx, Object msg) {
        if (this.state() == State.FAILED) {
            return;
        }
        LOG.debug("Negotiation read invoked on channel {}", (Object)this.channel);
        try {
            this.handleMessage((HelloMessage)msg);
        }
        catch (Exception e) {
            LOG.debug("Unexpected error while handling negotiation message {} on channel {}", new Object[]{msg, this.channel, e});
            this.negotiationFailed(e);
        }
    }

    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) {
        LOG.info("Unexpected error during negotiation on channel {}", (Object)this.channel, (Object)cause);
        this.negotiationFailed(cause);
    }

    protected abstract void handleMessage(HelloMessage var1) throws Exception;

    static {
        int propValue = Integer.getInteger(DEFAULT_MAXIMUM_CHUNK_SIZE_PROP, 0x1000000);
        if (propValue <= 0) {
            LOG.warn("Ignoring invalid {} value {}", (Object)DEFAULT_MAXIMUM_CHUNK_SIZE_PROP, (Object)propValue);
            DEFAULT_MAXIMUM_INCOMING_CHUNK_SIZE = 0x1000000;
        } else {
            DEFAULT_MAXIMUM_INCOMING_CHUNK_SIZE = propValue;
        }
        LOG.debug("Default maximum incoming NETCONF chunk size is {} bytes", (Object)DEFAULT_MAXIMUM_INCOMING_CHUNK_SIZE);
    }

    protected static enum State {
        IDLE,
        OPEN_WAIT,
        FAILED,
        ESTABLISHED;

    }
}

