/*
 * Decompiled with CFR 0.152.
 */
package org.opendaylight.yangtools.yang.data.codec.gson;

import com.google.common.annotations.Beta;
import com.google.common.base.Preconditions;
import com.google.gson.JsonIOException;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSyntaxException;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.MalformedJsonException;
import java.io.Closeable;
import java.io.EOFException;
import java.io.Flushable;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Deque;
import java.util.HashSet;
import java.util.Set;
import org.opendaylight.yangtools.yang.common.QName;
import org.opendaylight.yangtools.yang.data.api.schema.stream.NormalizedNodeStreamWriter;
import org.opendaylight.yangtools.yang.data.codec.gson.AbstractNodeDataWithSchema;
import org.opendaylight.yangtools.yang.data.codec.gson.AnyXmlNodeDataWithSchema;
import org.opendaylight.yangtools.yang.data.codec.gson.CompositeNodeDataWithSchema;
import org.opendaylight.yangtools.yang.data.codec.gson.JSONCodecFactory;
import org.opendaylight.yangtools.yang.data.codec.gson.LeafListEntryNodeDataWithSchema;
import org.opendaylight.yangtools.yang.data.codec.gson.LeafListNodeDataWithSchema;
import org.opendaylight.yangtools.yang.data.codec.gson.ListEntryNodeDataWithSchema;
import org.opendaylight.yangtools.yang.data.codec.gson.ListNodeDataWithSchema;
import org.opendaylight.yangtools.yang.data.codec.gson.SimpleNodeDataWithSchema;
import org.opendaylight.yangtools.yang.data.codec.gson.TopLevelNodeDataWithSchema;
import org.opendaylight.yangtools.yang.model.api.AnyXmlSchemaNode;
import org.opendaylight.yangtools.yang.model.api.ChoiceCaseNode;
import org.opendaylight.yangtools.yang.model.api.ChoiceNode;
import org.opendaylight.yangtools.yang.model.api.DataNodeContainer;
import org.opendaylight.yangtools.yang.model.api.DataSchemaNode;
import org.opendaylight.yangtools.yang.model.api.LeafListSchemaNode;
import org.opendaylight.yangtools.yang.model.api.LeafSchemaNode;
import org.opendaylight.yangtools.yang.model.api.Module;
import org.opendaylight.yangtools.yang.model.api.SchemaContext;
import org.opendaylight.yangtools.yang.model.api.SchemaNode;
import org.opendaylight.yangtools.yang.model.api.TypeDefinition;

@Beta
public final class JsonParserStream
implements Closeable,
Flushable {
    private final Deque<URI> namespaces = new ArrayDeque<URI>();
    private final NormalizedNodeStreamWriter writer;
    private final JSONCodecFactory codecs;
    private final SchemaContext schema;
    private final DataSchemaNode parentNode;

    private JsonParserStream(NormalizedNodeStreamWriter writer, SchemaContext schemaContext, DataSchemaNode parentNode) {
        this.schema = (SchemaContext)Preconditions.checkNotNull((Object)schemaContext);
        this.writer = (NormalizedNodeStreamWriter)Preconditions.checkNotNull((Object)writer);
        this.codecs = JSONCodecFactory.create(schemaContext);
        this.parentNode = parentNode;
    }

    public static JsonParserStream create(NormalizedNodeStreamWriter writer, SchemaContext schemaContext, SchemaNode parentNode) {
        Preconditions.checkArgument((boolean)(parentNode instanceof DataSchemaNode), (Object)"Instance of DataSchemaNode class awaited.");
        return new JsonParserStream(writer, schemaContext, (DataSchemaNode)parentNode);
    }

    public static JsonParserStream create(NormalizedNodeStreamWriter writer, SchemaContext schemaContext) {
        return new JsonParserStream(writer, schemaContext, (DataSchemaNode)schemaContext);
    }

    public JsonParserStream parse(JsonReader reader) throws JsonIOException, JsonSyntaxException {
        boolean lenient = reader.isLenient();
        reader.setLenient(true);
        boolean isEmpty = true;
        try {
            reader.peek();
            isEmpty = false;
            TopLevelNodeDataWithSchema topLevelNodeDataWithSchema = new TopLevelNodeDataWithSchema(this.parentNode);
            this.read(reader, topLevelNodeDataWithSchema);
            topLevelNodeDataWithSchema.normalizeTopLevelNode();
            topLevelNodeDataWithSchema.write(this.writer);
            JsonParserStream jsonParserStream = this;
            return jsonParserStream;
        }
        catch (EOFException e) {
            if (isEmpty) {
                JsonParserStream jsonParserStream = this;
                return jsonParserStream;
            }
            throw new JsonSyntaxException((Throwable)e);
        }
        catch (MalformedJsonException e) {
            throw new JsonSyntaxException((Throwable)e);
        }
        catch (IOException e) {
            throw new JsonIOException((Throwable)e);
        }
        catch (NumberFormatException e) {
            throw new JsonSyntaxException((Throwable)e);
        }
        catch (OutOfMemoryError | StackOverflowError e) {
            throw new JsonParseException("Failed parsing JSON source: " + reader + " to Json", (Throwable)e);
        }
        finally {
            reader.setLenient(lenient);
        }
    }

    private final void setValue(AbstractNodeDataWithSchema parent, String value) {
        Preconditions.checkArgument((boolean)(parent instanceof SimpleNodeDataWithSchema), (String)"Node %s is not a simple type", (Object[])new Object[]{parent});
        Object translatedValue = this.translateValueByType(value, parent.getSchema());
        ((SimpleNodeDataWithSchema)parent).setValue(translatedValue);
    }

    public void read(JsonReader in, AbstractNodeDataWithSchema parent) throws IOException {
        switch (in.peek()) {
            case STRING: 
            case NUMBER: {
                this.setValue(parent, in.nextString());
                break;
            }
            case BOOLEAN: {
                this.setValue(parent, Boolean.toString(in.nextBoolean()));
                break;
            }
            case NULL: {
                in.nextNull();
                this.setValue(parent, null);
                break;
            }
            case BEGIN_ARRAY: {
                in.beginArray();
                while (in.hasNext()) {
                    AbstractNodeDataWithSchema newChild = null;
                    if (parent instanceof ListNodeDataWithSchema) {
                        newChild = new ListEntryNodeDataWithSchema(parent.getSchema());
                        ((CompositeNodeDataWithSchema)parent).addChild(newChild);
                    } else if (parent instanceof LeafListNodeDataWithSchema) {
                        newChild = new LeafListEntryNodeDataWithSchema(parent.getSchema());
                        ((CompositeNodeDataWithSchema)parent).addChild(newChild);
                    }
                    this.read(in, newChild);
                }
                in.endArray();
                return;
            }
            case BEGIN_OBJECT: {
                HashSet<String> namesakes = new HashSet<String>();
                in.beginObject();
                while (in.hasNext()) {
                    String jsonElementName = in.nextName();
                    NamespaceAndName namespaceAndName = this.resolveNamespace(jsonElementName, parent.getSchema());
                    String localName = namespaceAndName.getName();
                    this.addNamespace(namespaceAndName.getUri());
                    if (namesakes.contains(jsonElementName)) {
                        throw new JsonSyntaxException("Duplicate name " + jsonElementName + " in JSON input.");
                    }
                    namesakes.add(jsonElementName);
                    Deque<DataSchemaNode> childDataSchemaNodes = this.findSchemaNodeByNameAndNamespace(parent.getSchema(), localName, this.getCurrentNamespace());
                    if (childDataSchemaNodes.isEmpty()) {
                        throw new IllegalStateException("Schema for node with name " + localName + " and namespace " + this.getCurrentNamespace() + " doesn't exist.");
                    }
                    AbstractNodeDataWithSchema newChild = ((CompositeNodeDataWithSchema)parent).addChild(childDataSchemaNodes);
                    if (newChild instanceof AnyXmlNodeDataWithSchema) {
                        in.skipValue();
                    } else {
                        this.read(in, newChild);
                    }
                    this.removeNamespace();
                }
                in.endObject();
                return;
            }
        }
    }

    private Object translateValueByType(String value, DataSchemaNode node) {
        TypeDefinition<? extends Object> typeDefinition = JsonParserStream.typeDefinition(node);
        if (typeDefinition == null) {
            return value;
        }
        return this.codecs.codecFor(typeDefinition).deserialize(value);
    }

    private static TypeDefinition<? extends Object> typeDefinition(DataSchemaNode node) {
        TypeDefinition baseType = null;
        if (node instanceof LeafListSchemaNode) {
            baseType = ((LeafListSchemaNode)node).getType();
        } else if (node instanceof LeafSchemaNode) {
            baseType = ((LeafSchemaNode)node).getType();
        } else {
            if (node instanceof AnyXmlSchemaNode) {
                return null;
            }
            throw new IllegalArgumentException("Unhandled parameter types: " + Arrays.asList(node).toString());
        }
        if (baseType != null) {
            while (baseType.getBaseType() != null) {
                baseType = baseType.getBaseType();
            }
        }
        return baseType;
    }

    private void removeNamespace() {
        this.namespaces.pop();
    }

    private void addNamespace(URI namespace) {
        this.namespaces.push(namespace);
    }

    private NamespaceAndName resolveNamespace(String childName, DataSchemaNode dataSchemaNode) {
        int lastIndexOfColon = childName.lastIndexOf(58);
        String moduleNamePart = null;
        String nodeNamePart = null;
        URI namespace = null;
        if (lastIndexOfColon != -1) {
            moduleNamePart = childName.substring(0, lastIndexOfColon);
            nodeNamePart = childName.substring(lastIndexOfColon + 1);
            Module m = this.schema.findModuleByName(moduleNamePart, null);
            namespace = m == null ? null : m.getNamespace();
        } else {
            nodeNamePart = childName;
        }
        if (namespace == null) {
            Set<Object> potentialUris = Collections.emptySet();
            potentialUris = this.resolveAllPotentialNamespaces(nodeNamePart, dataSchemaNode);
            if (potentialUris.contains(this.getCurrentNamespace())) {
                namespace = this.getCurrentNamespace();
            } else if (potentialUris.size() == 1) {
                namespace = (URI)potentialUris.iterator().next();
            } else {
                if (potentialUris.size() > 1) {
                    throw new IllegalStateException("Choose suitable module name for element " + nodeNamePart + ":" + this.toModuleNames(potentialUris));
                }
                if (potentialUris.isEmpty()) {
                    throw new IllegalStateException("Schema node with name " + nodeNamePart + " wasn't found.");
                }
            }
        }
        return new NamespaceAndName(nodeNamePart, namespace);
    }

    private String toModuleNames(Set<URI> potentialUris) {
        StringBuilder builder = new StringBuilder();
        for (URI potentialUri : potentialUris) {
            builder.append("\n");
            builder.append(((Module)this.schema.findModuleByNamespace(potentialUri).iterator().next()).getName());
        }
        return builder.toString();
    }

    private Set<URI> resolveAllPotentialNamespaces(String elementName, DataSchemaNode dataSchemaNode) {
        HashSet<URI> potentialUris = new HashSet<URI>();
        HashSet<ChoiceNode> choices = new HashSet<ChoiceNode>();
        if (dataSchemaNode instanceof DataNodeContainer) {
            for (DataSchemaNode childSchemaNode : ((DataNodeContainer)dataSchemaNode).getChildNodes()) {
                if (childSchemaNode instanceof ChoiceNode) {
                    choices.add((ChoiceNode)childSchemaNode);
                    continue;
                }
                if (!childSchemaNode.getQName().getLocalName().equals(elementName)) continue;
                potentialUris.add(childSchemaNode.getQName().getNamespace());
            }
            for (ChoiceNode choiceNode : choices) {
                for (ChoiceCaseNode concreteCase : choiceNode.getCases()) {
                    potentialUris.addAll(this.resolveAllPotentialNamespaces(elementName, (DataSchemaNode)concreteCase));
                }
            }
        }
        return potentialUris;
    }

    private URI getCurrentNamespace() {
        return this.namespaces.peek();
    }

    private Deque<DataSchemaNode> findSchemaNodeByNameAndNamespace(DataSchemaNode dataSchemaNode, String childName, URI namespace) {
        ArrayDeque<DataSchemaNode> result = new ArrayDeque<DataSchemaNode>();
        ArrayList<ChoiceNode> childChoices = new ArrayList<ChoiceNode>();
        if (dataSchemaNode instanceof DataNodeContainer) {
            for (DataSchemaNode childNode : ((DataNodeContainer)dataSchemaNode).getChildNodes()) {
                if (childNode instanceof ChoiceNode) {
                    childChoices.add((ChoiceNode)childNode);
                    continue;
                }
                QName childQName = childNode.getQName();
                if (!childQName.getLocalName().equals(childName) || !childQName.getNamespace().equals(namespace)) continue;
                result.push(childNode);
                return result;
            }
        }
        for (ChoiceNode choiceNode : childChoices) {
            for (ChoiceCaseNode concreteCase : choiceNode.getCases()) {
                Deque<DataSchemaNode> resultFromRecursion = this.findSchemaNodeByNameAndNamespace((DataSchemaNode)concreteCase, childName, namespace);
                if (resultFromRecursion.isEmpty()) continue;
                resultFromRecursion.push((DataSchemaNode)concreteCase);
                resultFromRecursion.push((DataSchemaNode)choiceNode);
                return resultFromRecursion;
            }
        }
        return result;
    }

    @Override
    public void flush() throws IOException {
        this.writer.flush();
    }

    @Override
    public void close() throws IOException {
        this.writer.flush();
        this.writer.close();
    }

    private static class NamespaceAndName {
        private final URI uri;
        private final String name;

        public NamespaceAndName(String name, URI uri) {
            this.name = name;
            this.uri = uri;
        }

        public String getName() {
            return this.name;
        }

        public URI getUri() {
            return this.uri;
        }
    }
}

