/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.net;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.base.CharMatcher;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.hash.Hashing;
import com.google.common.io.ByteStreams;
import com.google.common.primitives.Ints;
import java.math.BigInteger;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Locale;
import org.checkerframework.checker.nullness.qual.Nullable;

@Beta
@GwtIncompatible
public final class InetAddresses {
    private static final int IPV4_PART_COUNT = 4;
    private static final int IPV6_PART_COUNT = 8;
    private static final char IPV4_DELIMITER = '.';
    private static final char IPV6_DELIMITER = ':';
    private static final CharMatcher IPV4_DELIMITER_MATCHER = CharMatcher.is('.');
    private static final CharMatcher IPV6_DELIMITER_MATCHER = CharMatcher.is(':');
    private static final Inet4Address LOOPBACK4 = (Inet4Address)InetAddresses.forString("127.0.0.1");
    private static final Inet4Address ANY4 = (Inet4Address)InetAddresses.forString("0.0.0.0");

    private InetAddresses() {
    }

    private static Inet4Address getInet4Address(byte[] bytes2) {
        Preconditions.checkArgument(bytes2.length == 4, "Byte array has invalid length for an IPv4 address: %s != 4.", bytes2.length);
        return (Inet4Address)InetAddresses.bytesToInetAddress(bytes2);
    }

    public static InetAddress forString(String ipString) {
        byte[] addr2 = InetAddresses.ipStringToBytes(ipString);
        if (addr2 == null) {
            throw InetAddresses.formatIllegalArgumentException("'%s' is not an IP string literal.", ipString);
        }
        return InetAddresses.bytesToInetAddress(addr2);
    }

    public static boolean isInetAddress(String ipString) {
        return InetAddresses.ipStringToBytes(ipString) != null;
    }

    private static byte @Nullable [] ipStringToBytes(String ipString) {
        boolean hasColon = false;
        boolean hasDot = false;
        int percentIndex = -1;
        for (int i2 = 0; i2 < ipString.length(); ++i2) {
            char c = ipString.charAt(i2);
            if (c == '.') {
                hasDot = true;
                continue;
            }
            if (c == ':') {
                if (hasDot) {
                    return null;
                }
                hasColon = true;
                continue;
            }
            if (c == '%') {
                percentIndex = i2;
                break;
            }
            if (Character.digit(c, 16) != -1) continue;
            return null;
        }
        if (hasColon) {
            if (hasDot && (ipString = InetAddresses.convertDottedQuadToHex(ipString)) == null) {
                return null;
            }
            if (percentIndex != -1) {
                ipString = ipString.substring(0, percentIndex);
            }
            return InetAddresses.textToNumericFormatV6(ipString);
        }
        if (hasDot) {
            if (percentIndex != -1) {
                return null;
            }
            return InetAddresses.textToNumericFormatV4(ipString);
        }
        return null;
    }

    private static byte @Nullable [] textToNumericFormatV4(String ipString) {
        if (IPV4_DELIMITER_MATCHER.countIn(ipString) + 1 != 4) {
            return null;
        }
        byte[] bytes2 = new byte[4];
        int start2 = 0;
        for (int i2 = 0; i2 < 4; ++i2) {
            int end2 = ipString.indexOf(46, start2);
            if (end2 == -1) {
                end2 = ipString.length();
            }
            try {
                bytes2[i2] = InetAddresses.parseOctet(ipString, start2, end2);
            }
            catch (NumberFormatException ex) {
                return null;
            }
            start2 = end2 + 1;
        }
        return bytes2;
    }

    private static byte @Nullable [] textToNumericFormatV6(String ipString) {
        int delimiterCount = IPV6_DELIMITER_MATCHER.countIn(ipString);
        if (delimiterCount < 2 || delimiterCount > 8) {
            return null;
        }
        int partsSkipped = 8 - (delimiterCount + 1);
        boolean hasSkip = false;
        for (int i2 = 0; i2 < ipString.length() - 1; ++i2) {
            if (ipString.charAt(i2) != ':' || ipString.charAt(i2 + 1) != ':') continue;
            if (hasSkip) {
                return null;
            }
            hasSkip = true;
            ++partsSkipped;
            if (i2 == 0) {
                ++partsSkipped;
            }
            if (i2 != ipString.length() - 2) continue;
            ++partsSkipped;
        }
        if (ipString.charAt(0) == ':' && ipString.charAt(1) != ':') {
            return null;
        }
        if (ipString.charAt(ipString.length() - 1) == ':' && ipString.charAt(ipString.length() - 2) != ':') {
            return null;
        }
        if (hasSkip && partsSkipped <= 0) {
            return null;
        }
        if (!hasSkip && delimiterCount + 1 != 8) {
            return null;
        }
        ByteBuffer rawBytes = ByteBuffer.allocate(16);
        try {
            int start2 = 0;
            if (ipString.charAt(0) == ':') {
                start2 = 1;
            }
            while (start2 < ipString.length()) {
                int end2 = ipString.indexOf(58, start2);
                if (end2 == -1) {
                    end2 = ipString.length();
                }
                if (ipString.charAt(start2) == ':') {
                    for (int i3 = 0; i3 < partsSkipped; ++i3) {
                        rawBytes.putShort((short)0);
                    }
                } else {
                    rawBytes.putShort(InetAddresses.parseHextet(ipString, start2, end2));
                }
                start2 = end2 + 1;
            }
        }
        catch (NumberFormatException ex) {
            return null;
        }
        return rawBytes.array();
    }

    private static @Nullable String convertDottedQuadToHex(String ipString) {
        int lastColon = ipString.lastIndexOf(58);
        String initialPart = ipString.substring(0, lastColon + 1);
        String dottedQuad = ipString.substring(lastColon + 1);
        byte[] quad = InetAddresses.textToNumericFormatV4(dottedQuad);
        if (quad == null) {
            return null;
        }
        String penultimate = Integer.toHexString((quad[0] & 0xFF) << 8 | quad[1] & 0xFF);
        String ultimate = Integer.toHexString((quad[2] & 0xFF) << 8 | quad[3] & 0xFF);
        return new StringBuilder(1 + String.valueOf(initialPart).length() + String.valueOf(penultimate).length() + String.valueOf(ultimate).length()).append(initialPart).append(penultimate).append(":").append(ultimate).toString();
    }

    private static byte parseOctet(String ipString, int start2, int end2) {
        int length2 = end2 - start2;
        if (length2 <= 0 || length2 > 3) {
            throw new NumberFormatException();
        }
        if (length2 > 1 && ipString.charAt(start2) == '0') {
            throw new NumberFormatException();
        }
        int octet = 0;
        for (int i2 = start2; i2 < end2; ++i2) {
            octet *= 10;
            int digit = Character.digit(ipString.charAt(i2), 10);
            if (digit < 0) {
                throw new NumberFormatException();
            }
            octet += digit;
        }
        if (octet > 255) {
            throw new NumberFormatException();
        }
        return (byte)octet;
    }

    private static short parseHextet(String ipString, int start2, int end2) {
        int length2 = end2 - start2;
        if (length2 <= 0 || length2 > 4) {
            throw new NumberFormatException();
        }
        int hextet = 0;
        for (int i2 = start2; i2 < end2; ++i2) {
            hextet <<= 4;
            hextet |= Character.digit(ipString.charAt(i2), 16);
        }
        return (short)hextet;
    }

    private static InetAddress bytesToInetAddress(byte[] addr2) {
        try {
            return InetAddress.getByAddress(addr2);
        }
        catch (UnknownHostException e) {
            throw new AssertionError((Object)e);
        }
    }

    public static String toAddrString(InetAddress ip2) {
        Preconditions.checkNotNull(ip2);
        if (ip2 instanceof Inet4Address) {
            return ip2.getHostAddress();
        }
        Preconditions.checkArgument(ip2 instanceof Inet6Address);
        byte[] bytes2 = ip2.getAddress();
        int[] hextets = new int[8];
        for (int i2 = 0; i2 < hextets.length; ++i2) {
            hextets[i2] = Ints.fromBytes((byte)0, (byte)0, bytes2[2 * i2], bytes2[2 * i2 + 1]);
        }
        InetAddresses.compressLongestRunOfZeroes(hextets);
        return InetAddresses.hextetsToIPv6String(hextets);
    }

    private static void compressLongestRunOfZeroes(int[] hextets) {
        int bestRunStart = -1;
        int bestRunLength = -1;
        int runStart = -1;
        for (int i2 = 0; i2 < hextets.length + 1; ++i2) {
            if (i2 < hextets.length && hextets[i2] == 0) {
                if (runStart >= 0) continue;
                runStart = i2;
                continue;
            }
            if (runStart < 0) continue;
            int runLength = i2 - runStart;
            if (runLength > bestRunLength) {
                bestRunStart = runStart;
                bestRunLength = runLength;
            }
            runStart = -1;
        }
        if (bestRunLength >= 2) {
            Arrays.fill(hextets, bestRunStart, bestRunStart + bestRunLength, -1);
        }
    }

    private static String hextetsToIPv6String(int[] hextets) {
        StringBuilder buf = new StringBuilder(39);
        boolean lastWasNumber = false;
        for (int i2 = 0; i2 < hextets.length; ++i2) {
            boolean thisIsNumber;
            boolean bl = thisIsNumber = hextets[i2] >= 0;
            if (thisIsNumber) {
                if (lastWasNumber) {
                    buf.append(':');
                }
                buf.append(Integer.toHexString(hextets[i2]));
            } else if (i2 == 0 || lastWasNumber) {
                buf.append("::");
            }
            lastWasNumber = thisIsNumber;
        }
        return buf.toString();
    }

    public static String toUriString(InetAddress ip2) {
        if (ip2 instanceof Inet6Address) {
            String string2 = InetAddresses.toAddrString(ip2);
            return new StringBuilder(2 + String.valueOf(string2).length()).append("[").append(string2).append("]").toString();
        }
        return InetAddresses.toAddrString(ip2);
    }

    public static InetAddress forUriString(String hostAddr) {
        InetAddress addr2 = InetAddresses.forUriStringNoThrow(hostAddr);
        if (addr2 == null) {
            throw InetAddresses.formatIllegalArgumentException("Not a valid URI IP literal: '%s'", hostAddr);
        }
        return addr2;
    }

    private static @Nullable InetAddress forUriStringNoThrow(String hostAddr) {
        int expectBytes;
        String ipString;
        Preconditions.checkNotNull(hostAddr);
        if (hostAddr.startsWith("[") && hostAddr.endsWith("]")) {
            ipString = hostAddr.substring(1, hostAddr.length() - 1);
            expectBytes = 16;
        } else {
            ipString = hostAddr;
            expectBytes = 4;
        }
        byte[] addr2 = InetAddresses.ipStringToBytes(ipString);
        if (addr2 == null || addr2.length != expectBytes) {
            return null;
        }
        return InetAddresses.bytesToInetAddress(addr2);
    }

    public static boolean isUriInetAddress(String ipString) {
        return InetAddresses.forUriStringNoThrow(ipString) != null;
    }

    public static boolean isCompatIPv4Address(Inet6Address ip2) {
        if (!ip2.isIPv4CompatibleAddress()) {
            return false;
        }
        byte[] bytes2 = ip2.getAddress();
        return bytes2[12] != 0 || bytes2[13] != 0 || bytes2[14] != 0 || bytes2[15] != 0 && bytes2[15] != 1;
    }

    public static Inet4Address getCompatIPv4Address(Inet6Address ip2) {
        Preconditions.checkArgument(InetAddresses.isCompatIPv4Address(ip2), "Address '%s' is not IPv4-compatible.", (Object)InetAddresses.toAddrString(ip2));
        return InetAddresses.getInet4Address(Arrays.copyOfRange(ip2.getAddress(), 12, 16));
    }

    public static boolean is6to4Address(Inet6Address ip2) {
        byte[] bytes2 = ip2.getAddress();
        return bytes2[0] == 32 && bytes2[1] == 2;
    }

    public static Inet4Address get6to4IPv4Address(Inet6Address ip2) {
        Preconditions.checkArgument(InetAddresses.is6to4Address(ip2), "Address '%s' is not a 6to4 address.", (Object)InetAddresses.toAddrString(ip2));
        return InetAddresses.getInet4Address(Arrays.copyOfRange(ip2.getAddress(), 2, 6));
    }

    public static boolean isTeredoAddress(Inet6Address ip2) {
        byte[] bytes2 = ip2.getAddress();
        return bytes2[0] == 32 && bytes2[1] == 1 && bytes2[2] == 0 && bytes2[3] == 0;
    }

    public static TeredoInfo getTeredoInfo(Inet6Address ip2) {
        Preconditions.checkArgument(InetAddresses.isTeredoAddress(ip2), "Address '%s' is not a Teredo address.", (Object)InetAddresses.toAddrString(ip2));
        byte[] bytes2 = ip2.getAddress();
        Inet4Address server = InetAddresses.getInet4Address(Arrays.copyOfRange(bytes2, 4, 8));
        int flags2 = ByteStreams.newDataInput(bytes2, 8).readShort() & 0xFFFF;
        int port = ~ByteStreams.newDataInput(bytes2, 10).readShort() & 0xFFFF;
        byte[] clientBytes = Arrays.copyOfRange(bytes2, 12, 16);
        for (int i2 = 0; i2 < clientBytes.length; ++i2) {
            clientBytes[i2] = ~clientBytes[i2];
        }
        Inet4Address client = InetAddresses.getInet4Address(clientBytes);
        return new TeredoInfo(server, client, port, flags2);
    }

    public static boolean isIsatapAddress(Inet6Address ip2) {
        if (InetAddresses.isTeredoAddress(ip2)) {
            return false;
        }
        byte[] bytes2 = ip2.getAddress();
        if ((bytes2[8] | 3) != 3) {
            return false;
        }
        return bytes2[9] == 0 && bytes2[10] == 94 && bytes2[11] == -2;
    }

    public static Inet4Address getIsatapIPv4Address(Inet6Address ip2) {
        Preconditions.checkArgument(InetAddresses.isIsatapAddress(ip2), "Address '%s' is not an ISATAP address.", (Object)InetAddresses.toAddrString(ip2));
        return InetAddresses.getInet4Address(Arrays.copyOfRange(ip2.getAddress(), 12, 16));
    }

    public static boolean hasEmbeddedIPv4ClientAddress(Inet6Address ip2) {
        return InetAddresses.isCompatIPv4Address(ip2) || InetAddresses.is6to4Address(ip2) || InetAddresses.isTeredoAddress(ip2);
    }

    public static Inet4Address getEmbeddedIPv4ClientAddress(Inet6Address ip2) {
        if (InetAddresses.isCompatIPv4Address(ip2)) {
            return InetAddresses.getCompatIPv4Address(ip2);
        }
        if (InetAddresses.is6to4Address(ip2)) {
            return InetAddresses.get6to4IPv4Address(ip2);
        }
        if (InetAddresses.isTeredoAddress(ip2)) {
            return InetAddresses.getTeredoInfo(ip2).getClient();
        }
        throw InetAddresses.formatIllegalArgumentException("'%s' has no embedded IPv4 address.", InetAddresses.toAddrString(ip2));
    }

    public static boolean isMappedIPv4Address(String ipString) {
        byte[] bytes2 = InetAddresses.ipStringToBytes(ipString);
        if (bytes2 != null && bytes2.length == 16) {
            int i2;
            for (i2 = 0; i2 < 10; ++i2) {
                if (bytes2[i2] == 0) continue;
                return false;
            }
            for (i2 = 10; i2 < 12; ++i2) {
                if (bytes2[i2] == -1) continue;
                return false;
            }
            return true;
        }
        return false;
    }

    public static Inet4Address getCoercedIPv4Address(InetAddress ip2) {
        if (ip2 instanceof Inet4Address) {
            return (Inet4Address)ip2;
        }
        byte[] bytes2 = ip2.getAddress();
        boolean leadingBytesOfZero = true;
        for (int i2 = 0; i2 < 15; ++i2) {
            if (bytes2[i2] == 0) continue;
            leadingBytesOfZero = false;
            break;
        }
        if (leadingBytesOfZero && bytes2[15] == 1) {
            return LOOPBACK4;
        }
        if (leadingBytesOfZero && bytes2[15] == 0) {
            return ANY4;
        }
        Inet6Address ip6 = (Inet6Address)ip2;
        long addressAsLong = 0L;
        addressAsLong = InetAddresses.hasEmbeddedIPv4ClientAddress(ip6) ? (long)InetAddresses.getEmbeddedIPv4ClientAddress(ip6).hashCode() : ByteBuffer.wrap(ip6.getAddress(), 0, 8).getLong();
        int coercedHash = Hashing.murmur3_32().hashLong(addressAsLong).asInt();
        if ((coercedHash |= 0xE0000000) == -1) {
            coercedHash = -2;
        }
        return InetAddresses.getInet4Address(Ints.toByteArray(coercedHash));
    }

    public static int coerceToInteger(InetAddress ip2) {
        return ByteStreams.newDataInput(InetAddresses.getCoercedIPv4Address(ip2).getAddress()).readInt();
    }

    public static BigInteger toBigInteger(InetAddress address2) {
        return new BigInteger(1, address2.getAddress());
    }

    public static Inet4Address fromInteger(int address2) {
        return InetAddresses.getInet4Address(Ints.toByteArray(address2));
    }

    public static Inet4Address fromIPv4BigInteger(BigInteger address2) {
        return (Inet4Address)InetAddresses.fromBigInteger(address2, false);
    }

    public static Inet6Address fromIPv6BigInteger(BigInteger address2) {
        return (Inet6Address)InetAddresses.fromBigInteger(address2, true);
    }

    private static InetAddress fromBigInteger(BigInteger address2, boolean isIpv6) {
        Preconditions.checkArgument(address2.signum() >= 0, "BigInteger must be greater than or equal to 0");
        int numBytes = isIpv6 ? 16 : 4;
        byte[] addressBytes = address2.toByteArray();
        byte[] targetCopyArray = new byte[numBytes];
        int srcPos = Math.max(0, addressBytes.length - numBytes);
        int copyLength = addressBytes.length - srcPos;
        int destPos = numBytes - copyLength;
        for (int i2 = 0; i2 < srcPos; ++i2) {
            if (addressBytes[i2] == 0) continue;
            throw InetAddresses.formatIllegalArgumentException("BigInteger cannot be converted to InetAddress because it has more than %d bytes: %s", numBytes, address2);
        }
        System.arraycopy(addressBytes, srcPos, targetCopyArray, destPos, copyLength);
        try {
            return InetAddress.getByAddress(targetCopyArray);
        }
        catch (UnknownHostException impossible) {
            throw new AssertionError((Object)impossible);
        }
    }

    public static InetAddress fromLittleEndianByteArray(byte[] addr2) throws UnknownHostException {
        byte[] reversed = new byte[addr2.length];
        for (int i2 = 0; i2 < addr2.length; ++i2) {
            reversed[i2] = addr2[addr2.length - i2 - 1];
        }
        return InetAddress.getByAddress(reversed);
    }

    public static InetAddress decrement(InetAddress address2) {
        int i2;
        byte[] addr2 = address2.getAddress();
        for (i2 = addr2.length - 1; i2 >= 0 && addr2[i2] == 0; --i2) {
            addr2[i2] = -1;
        }
        Preconditions.checkArgument(i2 >= 0, "Decrementing %s would wrap.", (Object)address2);
        int n = i2;
        addr2[n] = (byte)(addr2[n] - 1);
        return InetAddresses.bytesToInetAddress(addr2);
    }

    public static InetAddress increment(InetAddress address2) {
        int i2;
        byte[] addr2 = address2.getAddress();
        for (i2 = addr2.length - 1; i2 >= 0 && addr2[i2] == -1; --i2) {
            addr2[i2] = 0;
        }
        Preconditions.checkArgument(i2 >= 0, "Incrementing %s would wrap.", (Object)address2);
        int n = i2;
        addr2[n] = (byte)(addr2[n] + 1);
        return InetAddresses.bytesToInetAddress(addr2);
    }

    public static boolean isMaximum(InetAddress address2) {
        byte[] addr2 = address2.getAddress();
        for (int i2 = 0; i2 < addr2.length; ++i2) {
            if (addr2[i2] == -1) continue;
            return false;
        }
        return true;
    }

    private static IllegalArgumentException formatIllegalArgumentException(String format, Object ... args2) {
        return new IllegalArgumentException(String.format(Locale.ROOT, format, args2));
    }

    @Beta
    public static final class TeredoInfo {
        private final Inet4Address server;
        private final Inet4Address client;
        private final int port;
        private final int flags;

        public TeredoInfo(@Nullable Inet4Address server, @Nullable Inet4Address client, int port, int flags2) {
            Preconditions.checkArgument(port >= 0 && port <= 65535, "port '%s' is out of range (0 <= port <= 0xffff)", port);
            Preconditions.checkArgument(flags2 >= 0 && flags2 <= 65535, "flags '%s' is out of range (0 <= flags <= 0xffff)", flags2);
            this.server = MoreObjects.firstNonNull(server, ANY4);
            this.client = MoreObjects.firstNonNull(client, ANY4);
            this.port = port;
            this.flags = flags2;
        }

        public Inet4Address getServer() {
            return this.server;
        }

        public Inet4Address getClient() {
            return this.client;
        }

        public int getPort() {
            return this.port;
        }

        public int getFlags() {
            return this.flags;
        }
    }
}

