/*
 * MIT License
 *
 * Copyright (c) 2021 OpeningO Co.,Ltd.
 *
 *    https://openingo.org
 *    contactus(at)openingo.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package org.openingo.spring.gedid.engine.zookeeper;

import lombok.extern.slf4j.Slf4j;
import org.apache.curator.framework.CuratorFramework;
import org.apache.zookeeper.data.Stat;
import org.openingo.spring.gedid.engine.IDidEngine;

import java.nio.charset.StandardCharsets;
import java.util.Objects;

/**
 * ZookeeperIdEngine
 *
 * @author Qicz
 * @since 2021/6/25 10:41
 */
@Slf4j
public class ZookeeperIdEngine implements IDidEngine<Long> {

	private final CuratorFramework curator;
	private Long startId;

	public ZookeeperIdEngine(CuratorFramework curator) {
		this.curator = curator;
	}

	@Override
	public void follow(String name) {
		this.follow(name, 0L);
	}

	@Override
	public void follow(String name, Long startId) {
		this.startId = startId;
		String path = this.getPath(name);
		try {
			Stat stat = this.curator.checkExists().forPath(path);
			if (Objects.isNull(stat)) {
				this.curator.create().forPath(path);
			}
		} catch (Exception e) {
			e.printStackTrace();
			log.error(e.getLocalizedMessage());
		}
	}

	@Override
	public Long next(String name) {
		String path = this.getPath(name);
		long id = -1L;
		try {
			Stat stat = this.curator.setData().forPath(path, ("").getBytes(StandardCharsets.UTF_8));
			id = (stat.getMzxid() - stat.getCzxid()) + this.startId;
		} catch (Exception e) {
			e.printStackTrace();
			log.error(e.getLocalizedMessage());
		}
		return id;
	}

	private String getPath(String name) {
		return String.format("/%s", name);
	}
}
