package org.openl.rules.datatype.binding;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import net.sf.cglib.core.ReflectUtils;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.CodeVisitor;
import org.objectweb.asm.Constants;
import org.objectweb.asm.Label;
import org.objectweb.asm.Type;
import org.openl.binding.MethodUtil;
import org.openl.types.IOpenClass;
import org.openl.types.IOpenField;
import org.openl.util.StringTool;
import org.openl.util.generation.JavaClassGeneratorHelper;

/**
 * Generates byte code for simple java bean.
 * 
 * @author DLiauchuk
 *
 */
public class SimpleBeanByteCodeGenerator {
    

    private static final String JAVA_LANG_OBJECT = "java/lang/Object";

    private final Log LOG = LogFactory.getLog(SimpleBeanByteCodeGenerator.class);
    
    private String beanName;
    private IOpenClass parentClass;
    private Map<String, FieldType> beanFields;
    private Map<String, FieldType> allFields;
    /**
     * Number of fields that will take 2 stack elements(like a double and long)
     */
    private int twoStackElementFieldsCount;
    private byte[] generatedByteCode;

    private String beanNameWithPackage;
    
    /**
     * 
     * @param beanName name of the generated class, with namespace (e.g. <code>my.test.TestClass</code>)
     * @param beanFields map of fields, field name as a key, and type as value.
     */
    public SimpleBeanByteCodeGenerator(String beanName, Map<String, FieldType> beanFields) {
        this(beanName, beanFields, null);
    }

    /**
     * 
     * @param beanName name of the generated class, with namespace (e.g.
     *            <code>my.test.TestClass</code>)
     * @param beanFields map of fields, field name as a key, and type as value.
     * @param parentClass parent class
     */
    public SimpleBeanByteCodeGenerator(String beanName, Map<String, FieldType> beanFields, IOpenClass parentClass) {
        this.beanName = beanName;
        this.beanFields = beanFields;
        this.parentClass = parentClass;
        allFields = new LinkedHashMap<String, FieldType>();
        if (parentClass != null) {
            allFields.putAll(convertFields(parentClass.getFields()));
        }
        allFields.putAll(beanFields);
        twoStackElementFieldsCount = getTwoStackElementFieldsCount(allFields);
    }
    
    /**
     *  
     * @return generated byte code of the simple bean
     */
    public byte[] generateClassByteCode() {
        beanNameWithPackage = JavaClassGeneratorHelper.replaceCommas(beanName);
        
        ClassWriter classWriter = new ClassWriter(false);

        writeClassDescription(beanNameWithPackage, classWriter);

        writeFields(classWriter);
        
        writeDefaultConstructor(classWriter);
        writeConstructorWithFields(classWriter);
        
        writeGettersAndSetters(classWriter);
        writeToStringMethod(classWriter);
        writeEqualsMethod(classWriter);
        writeHashCodeMethod(classWriter);
        
        generatedByteCode = classWriter.toByteArray();
        
//        writeBytesToFile(generatedByteCode);
        
        return generatedByteCode;
    }
    
    /**
     * 
     * @return <code>Class<?></code> object of the generated bean class.
     */
    public Class<?> generateAndLoadBeanClass() {
        if (generatedByteCode != null) {
            return loadClass(generatedByteCode);
        } else {
            return loadClass(generateClassByteCode());
        }
    }
    
    /**
     * Writes getters and setters to the generated bean class.
     * 
     * @param beanNameWithPackage
     * @param classWriter
     */
    private void writeGettersAndSetters(ClassWriter classWriter) {
        for(Map.Entry<String, FieldType> field : beanFields.entrySet()) {
            generateGetter(beanNameWithPackage, classWriter, field);
            generateSetter(beanNameWithPackage, classWriter, field);
        }
    }
    
    private void writeToStringMethod(ClassWriter classWriter) {
        CodeVisitor codeVisitor;
        codeVisitor = classWriter.visitMethod(Constants.ACC_PUBLIC, "toString", String.format("()%s",
                getJavaType(String.class)), null, null);

        // create StringBuilder
        codeVisitor.visitTypeInsn(Constants.NEW, Type.getInternalName(StringBuilder.class));
        codeVisitor.visitInsn(Constants.DUP);
        codeVisitor
                .visitMethodInsn(Constants.INVOKESPECIAL, Type.getInternalName(StringBuilder.class), "<init>", "()V");

        // write ClassName
        codeVisitor.visitVarInsn(Constants.ALOAD, 0);
        invokeVirtual(codeVisitor, Object.class, "getClass", new Class<?>[] {});
        invokeVirtual(codeVisitor, Class.class, "getSimpleName", new Class<?>[] {});
        invokeVirtual(codeVisitor, StringBuilder.class, "append", new Class<?>[] { String.class });

        // write fields
        codeVisitor.visitLdcInsn("{ ");
        invokeVirtual(codeVisitor, StringBuilder.class, "append", new Class<?>[] { String.class });
        for (Map.Entry<String, FieldType> field : allFields.entrySet()) {
            codeVisitor.visitLdcInsn(field.getKey() + "=");
            invokeVirtual(codeVisitor, StringBuilder.class, "append", new Class<?>[] { String.class });

            pushFieldToStack(codeVisitor, 0, field.getKey());
            if (field.getValue().isArray()) { 
                invokeStatic(codeVisitor, ArrayUtils.class, "toString", new Class<?>[] { getJavaClass(field.getValue()) });
            }
            invokeVirtual(codeVisitor, StringBuilder.class, "append", new Class<?>[] { getJavaClass(field.getValue()) });

            codeVisitor.visitLdcInsn(" ");
            invokeVirtual(codeVisitor, StringBuilder.class, "append", new Class<?>[] { String.class });
        }
        codeVisitor.visitLdcInsn("}");
        invokeVirtual(codeVisitor, StringBuilder.class, "append", new Class<?>[] { String.class });

        // return
        invokeVirtual(codeVisitor, StringBuilder.class, "toString", new Class<?>[] {});
        codeVisitor.visitInsn(getConstantForReturn(String.class));
        if (twoStackElementFieldsCount > 0) {
            codeVisitor.visitMaxs(3, 1);
        } else {
            codeVisitor.visitMaxs(2, 1);
        }
    }

    private void writeEqualsMethod(ClassWriter classWriter) {
        CodeVisitor codeVisitor;
        codeVisitor = classWriter.visitMethod(Constants.ACC_PUBLIC, "equals", String.format("(%s)%s",
                getJavaType(Object.class), getJavaType(boolean.class)), null, null);

        // create EqualsBuilder
        codeVisitor.visitTypeInsn(Constants.NEW, Type.getInternalName(EqualsBuilder.class));
        codeVisitor.visitInsn(Constants.DUP);
        codeVisitor
                .visitMethodInsn(Constants.INVOKESPECIAL, Type.getInternalName(EqualsBuilder.class), "<init>", "()V");

        Label comparingLabel = new Label();

        // check "instance of" object
        codeVisitor.visitVarInsn(Constants.ALOAD, 1);
        codeVisitor.visitTypeInsn(Constants.INSTANCEOF, beanNameWithPackage);
        codeVisitor.visitJumpInsn(Constants.IFNE, comparingLabel);
        codeVisitor.visitLdcInsn(Boolean.FALSE);
        codeVisitor.visitInsn(getConstantForReturn(boolean.class));

        // cast
        codeVisitor.visitLabel(comparingLabel);
        codeVisitor.visitVarInsn(Constants.ALOAD, 1);
        codeVisitor.visitTypeInsn(Constants.CHECKCAST, beanNameWithPackage);
        codeVisitor.visitVarInsn(Constants.ASTORE, 2);

        // comparing by fields
        for (Map.Entry<String, FieldType> field : allFields.entrySet()) {
            pushFieldToStack(codeVisitor, 0, field.getKey());
            pushFieldToStack(codeVisitor, 2, field.getKey());

            Class<?> fieldType = getJavaClass(field.getValue());
            invokeVirtual(codeVisitor, EqualsBuilder.class, "append", new Class<?>[] { fieldType, fieldType });
        }

        invokeVirtual(codeVisitor, EqualsBuilder.class, "isEquals", new Class<?>[] {});

        codeVisitor.visitInsn(getConstantForReturn(boolean.class));
        if (twoStackElementFieldsCount > 0) {
            codeVisitor.visitMaxs(5, 3);
        } else {
            codeVisitor.visitMaxs(3, 3);
        }
    }

    private void writeHashCodeMethod(ClassWriter classWriter) {
        CodeVisitor codeVisitor;
        codeVisitor = classWriter.visitMethod(Constants.ACC_PUBLIC, "hashCode", String.format("()%s",
                getJavaType(int.class)), null, null);

        // create HashCodeBuilder
        codeVisitor.visitTypeInsn(Constants.NEW, Type.getInternalName(HashCodeBuilder.class));
        codeVisitor.visitInsn(Constants.DUP);
        codeVisitor.visitMethodInsn(Constants.INVOKESPECIAL, Type.getInternalName(HashCodeBuilder.class), "<init>",
                "()V");

        // generating hash code by fields
        for (Map.Entry<String, FieldType> field : allFields.entrySet()) {
            pushFieldToStack(codeVisitor, 0, field.getKey());
            invokeVirtual(codeVisitor, HashCodeBuilder.class, "append",
                    new Class<?>[] { getJavaClass(field.getValue()) });
        }
        invokeVirtual(codeVisitor, HashCodeBuilder.class, "toHashCode", new Class<?>[] {});

        codeVisitor.visitInsn(getConstantForReturn(int.class));
        if (twoStackElementFieldsCount > 0) {
            codeVisitor.visitMaxs(3, 1);
        } else {
            codeVisitor.visitMaxs(2, 2);
        }
    }
    
    private void writeConstructorWithFields(ClassWriter classWriter) {
        CodeVisitor codeVisitor;
        
        Constructor<?> parentConstructorWithFields = null;
        if(parentClass != null){
            parentConstructorWithFields = JavaClassGeneratorHelper.getBeanConstructorWithAllFields(parentClass
                    .getInstanceClass(), parentClass.getFields().size());
        }
        int i = 1;
        int stackSizeForParentConstructorCall = 0;
        if(parentConstructorWithFields == null){
            codeVisitor = classWriter.visitMethod(Constants.ACC_PUBLIC, "<init>", getMethodSignatureForByteCode(
                    beanFields, null), null, null);
            codeVisitor.visitVarInsn(Constants.ALOAD, 0);
            if (parentClass == null) {
                codeVisitor.visitMethodInsn(Constants.INVOKESPECIAL, JAVA_LANG_OBJECT, "<init>", "()V");
            }else{
                codeVisitor.visitMethodInsn(Constants.INVOKESPECIAL, Type.getInternalName(parentClass
                        .getInstanceClass()), "<init>", "()V");
            }
        }else{
            codeVisitor = classWriter.visitMethod(Constants.ACC_PUBLIC, "<init>", getMethodSignatureForByteCode(
                    allFields, null), null, null);
            codeVisitor.visitVarInsn(Constants.ALOAD, 0);

            Map<String, FieldType> parentFields = convertFields(parentClass.getFields());

            // push to stack all parameters for parent constructor
            for (Map.Entry<String, FieldType> field : parentFields.entrySet()) {
                FieldType fieldType = field.getValue();
                codeVisitor.visitVarInsn(getConstantForVarInsn(fieldType), i);
                if (long.class.equals(fieldType.getType()) || double.class.equals(fieldType.getType())) {
                    i += 2;
                } else {
                    i++;
                }
            }

            // invoke parent constructor with fields
            stackSizeForParentConstructorCall = i;
            codeVisitor.visitMethodInsn(Constants.INVOKESPECIAL, Type.getInternalName(parentClass.getInstanceClass()),
                    "<init>", getMethodSignatureForByteCode(parentFields, null));
        }

        // set all fields that is not presented in parent
        for (Map.Entry<String, FieldType> field : beanFields.entrySet()) {
            String fieldName = field.getKey();
            if (parentClass == null || parentClass.getField(fieldName) == null) {
                // there is no such field in parent class
                FieldType fieldType = field.getValue();
                codeVisitor.visitVarInsn(Constants.ALOAD, 0);
                codeVisitor.visitVarInsn(getConstantForVarInsn(fieldType), i);
                codeVisitor.visitFieldInsn(Constants.PUTFIELD, beanNameWithPackage, fieldName, getJavaType(fieldType));
                if (long.class.equals(fieldType.getType()) || double.class.equals(fieldType.getType())) {
                    i += 2;
                } else {
                    i++;
                }
            }
        }

        codeVisitor.visitInsn(Constants.RETURN);
        if (twoStackElementFieldsCount > 0) {
            codeVisitor.visitMaxs(3 + stackSizeForParentConstructorCall, allFields.size() + 1
                    + twoStackElementFieldsCount);
        } else {
            codeVisitor.visitMaxs(2 + stackSizeForParentConstructorCall, allFields.size() + 1);
        }
    }
    
    private void writeDefaultConstructor(ClassWriter classWriter) {
        CodeVisitor codeVisitor;
        // creates a MethodWriter for the (implicit) constructor
        codeVisitor = classWriter.visitMethod(Constants.ACC_PUBLIC, "<init>", "()V", null, null);
        // pushes the 'this' variable
        codeVisitor.visitVarInsn(Constants.ALOAD, 0);
        // invokes the super class constructor
        if (parentClass == null) {
            codeVisitor.visitMethodInsn(Constants.INVOKESPECIAL, JAVA_LANG_OBJECT, "<init>", "()V");
        } else {
            codeVisitor.visitMethodInsn(Constants.INVOKESPECIAL, Type.getInternalName(parentClass.getInstanceClass()),
                    "<init>", "()V");
        }
        codeVisitor.visitInsn(Constants.RETURN);
        // this code uses a maximum of one stack element and one local variable
        codeVisitor.visitMaxs(1, 1);
    }
    
    /**
     * Write fields declarations to the generated bean class.
     * 
     * @param classWriter
     */
    private void writeFields(ClassWriter classWriter) {
        for (Map.Entry<String,  FieldType> field : beanFields.entrySet()) {
          String fieldTypeName = getJavaType(field.getValue());
          classWriter.visitField(Constants.ACC_PROTECTED, field.getKey(), fieldTypeName, null, null);
        }
    }
    
    /**
     * Write the description of generating class.
     * 
     * @param beanNameWithPackage name of the class with package, symbol '/' is used as separator<br> 
     * (e.g. <code>my/test/TestClass</code>) 
     * @param classWriter
     */
    private void writeClassDescription(String beanNameWithPackage, ClassWriter classWriter) {
        String sourceFileName = JavaClassGeneratorHelper.getClassFileName((beanNameWithPackage));
        if (parentClass == null) {
            classWriter.visit(Constants.V1_5, Constants.ACC_PUBLIC + Constants.ACC_SUPER, beanNameWithPackage,
                    JAVA_LANG_OBJECT, null, sourceFileName);
        } else {
            classWriter.visit(Constants.V1_5, Constants.ACC_PUBLIC + Constants.ACC_SUPER, beanNameWithPackage, Type
                    .getInternalName(parentClass.getInstanceClass()), null, sourceFileName);
        }
    }

    @SuppressWarnings("unused")
    private void writeBytesToFile(byte[] byteArray) {
        String strFilePath = String.format("D://%s.class", JavaClassGeneratorHelper.getShortClassName(beanName));
        try {
            FileOutputStream fos = new FileOutputStream(strFilePath);
            fos.write(byteArray);
            fos.close();
        } catch(FileNotFoundException ex) {
            LOG.error(this, ex);
        } catch(IOException ioe){
            LOG.error(this, ioe);
        }
    }
    
    /**
     * Generates getter for the field.
     * 
     * @param beanNameWithPackage
     * @param classWriter
     * @param field
     */
    private void generateGetter(String beanNameWithPackage, ClassWriter classWriter, Map.Entry<String, FieldType> field) {
        CodeVisitor codeVisitor;
        String fieldName = field.getKey();
        FieldType fieldType = field.getValue();
        String getterName = StringTool.getGetterName(fieldName);
        
        codeVisitor = classWriter.visitMethod(Constants.ACC_PUBLIC,  getterName, String.format("()%s",
            getJavaType(fieldType)), null, null);
        codeVisitor.visitVarInsn(Constants.ALOAD, 0);
        codeVisitor.visitFieldInsn(Constants.GETFIELD, beanNameWithPackage, fieldName, getJavaType(fieldType));
        codeVisitor.visitInsn(getConstantForReturn(fieldType));
        
        // long and double types are the biggest ones, so they use a maximum of two stack  
        // elements and one local variable for getter method.
        if (long.class.equals(fieldType.getType()) || double.class.equals(fieldType.getType())) {
            codeVisitor.visitMaxs(2, 1);
        } else {
            codeVisitor.visitMaxs(1, 1);
        }
    }
    
    private int getConstantForReturn(FieldType fieldType) {
        Class<?> retClass = fieldType.getType();
        if (retClass != null) {
            return getConstantForReturn(retClass);
        } else {
            return Constants.ARETURN;
        }
    }
    
    /**
     * Returns the constant for return type. Each primitive type has its constant.
     * 
     * @param fieldClass
     * @return
     */
    private int getConstantForReturn(Class<?> fieldClass) {
        if (fieldClass.equals(int.class) || fieldClass.equals(short.class) || fieldClass.equals(boolean.class)
                || fieldClass.equals(char.class) || fieldClass.equals(byte.class)) {
            return Constants.IRETURN;
        } else if (fieldClass.equals(long.class)) {
            return Constants.LRETURN;
        } else if (fieldClass.equals(double.class)) {
            return Constants.DRETURN;
        } else if (fieldClass.equals(float.class)) {
            return Constants.FRETURN;
        } else if (fieldClass instanceof Object) {
            return Constants.ARETURN;
        }
        return 0;
    }
    
    /**
     * Generates setter for the field.
     * 
     * @param beanNameWithPackage
     * @param classWriter
     * @param field
     */
    private void generateSetter(String beanNameWithPackage, ClassWriter classWriter, Map.Entry<String, FieldType> field) {
        CodeVisitor codeVisitor;
        String fieldName = field.getKey();
        FieldType fieldType = field.getValue();        
        String setterName = StringTool.getSetterName(field.getKey());
        
        codeVisitor = classWriter.visitMethod(Constants.ACC_PUBLIC,  setterName, String.format("(%s)V", 
            getJavaType(fieldType)), null, null);
        codeVisitor.visitVarInsn(Constants.ALOAD, 0);
        codeVisitor.visitVarInsn(getConstantForVarInsn(fieldType), 1);
        
        codeVisitor.visitFieldInsn(Constants.PUTFIELD, beanNameWithPackage, fieldName, getJavaType(fieldType));
        codeVisitor.visitInsn(Constants.RETURN);
        
        // long and double types are the biggest ones, so they use a maximum of three stack  
        // elements and three local variables for setter method.
        if (long.class.equals(fieldType.getType()) || double.class.equals(fieldType.getType())) {
            codeVisitor.visitMaxs(3, 3);
        } else {
            codeVisitor.visitMaxs(2, 2);
        }
    }
 
    private int getConstantForVarInsn(FieldType fieldType) {
        Class<?> retClass = fieldType.getType();
        if (retClass != null) {
            return getConstantForVarInsn(retClass);
        } else {
            return Constants.ALOAD;
        }
    }

    private int getConstantForVarInsn(Class<?> fieldClass) {
        if (fieldClass.equals(int.class) || fieldClass.equals(short.class) || fieldClass.equals(boolean.class)
                || fieldClass.equals(char.class) || fieldClass.equals(byte.class)) {
            return Constants.ILOAD;        
        } else if (fieldClass.equals(long.class)) {
            return Constants.LLOAD;
        } else if (fieldClass.equals(double.class)) {
            return Constants.DLOAD;
        } else if (fieldClass.equals(float.class)) {
            return Constants.FLOAD;
        } else if (fieldClass instanceof Object) {
            return Constants.ALOAD;
        }
        return 0;
    }   
    
    /**
     * Return loaded to classpath class object
     * 
     * @param byteCode generated byteCode
     * 
     * @return <code>Class<?></code> descriptor for given byteCode
     */
    private Class<?> loadClass(byte[] byteCode) {
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        try {
            // try to define bean class in classloader, and return
            // class object.
            return ReflectUtils.defineClass(beanName, byteCode, classLoader);
        } catch (Exception ex) {
            LOG.debug(this, ex);
            try {
                // if defining fails, when this class already exists in
                // classloader,
                // try to return class object previosly loaded.
                return Class.forName(beanName, true, classLoader);
            } catch (Exception e) {
                LOG.error(this, e);
            } catch (VerifyError exc) {
                LOG.error(this, exc);
            }
        }

        return null;
    }
    
    private Class<?> getJavaClass(FieldType fieldType) {
        if (fieldType.getType() == null) {
            return Object.class;
        } else {
            return fieldType.getType();
        }
    }
    
    /**
     * Returns the Java type corresponding to the given class.
     * 
     * @param fieldClass
     * @return the Java type corresponding to the given class.
     */
    private String getJavaType(Class<?> fieldClass) {
        return String.valueOf(Type.getType(fieldClass));
    }
    
    /**
     * Gets Java type corresponding to the given field type.
     * 
     * @param fieldType
     * @return Java type corresponding to the given field type. (e.g. <code>Lmy/test/TestClass;</code>)
     */
    private String getJavaType(FieldType fieldType) {
        Class<?> fieldClass = fieldType.getType();
        if (fieldClass != null) {
            return getJavaType(fieldClass);
        } else {
            return JavaClassGeneratorHelper.getJavaType(fieldType.getTypeName());
        }
    }

    private Map<String, FieldType> convertFields(Map<String, IOpenField> fieldsToConvert) {
        Map<String, FieldType> fields = new LinkedHashMap<String, FieldType>();
        for (Entry<String, IOpenField> field : fieldsToConvert.entrySet()) {
            fields.put(field.getKey(), new FieldType(field.getValue()));
        }
        return fields;
    }

    private int getTwoStackElementFieldsCount(Map<String, FieldType> fields) {
        int twoStackElementsCount = 0;
        for (FieldType fieldType : fields.values()) {
            if (long.class.equals(fieldType.getType()) || double.class.equals(fieldType.getType())) {
                twoStackElementsCount++;
            }
        }
        return twoStackElementsCount;
    }

    private void invokeVirtual(CodeVisitor codeVisitor, Class<?> methodOwner, String methodName, Class<?>[] paramTypes) {
        String signatureBuilder = getSignature(methodOwner, methodName, paramTypes);        
        codeVisitor.visitMethodInsn(Constants.INVOKEVIRTUAL, Type.getInternalName(methodOwner), methodName, signatureBuilder);
    }
    
    private void invokeStatic(CodeVisitor codeVisitor, Class<?> methodOwner, String methodName, Class<?>[] paramTypes) {        
        String signatureBuilder = getSignature(methodOwner, methodName, paramTypes);
        codeVisitor.visitMethodInsn(Constants.INVOKESTATIC, Type.getInternalName(methodOwner), methodName, signatureBuilder);
    }

    private String getMethodSignatureForByteCode(Map<String, FieldType> params, Class<?> returnType){
        StringBuilder signatureBuilder = new StringBuilder("(");
        for (Map.Entry<String, FieldType> field : params.entrySet()) {
            String javaType = getJavaType(field.getValue());
            signatureBuilder.append(javaType);
        }
        signatureBuilder.append(")");
        if(returnType == null){
            signatureBuilder.append("V");
        }else{
            signatureBuilder.append(getJavaType(returnType));
        }
        return signatureBuilder.toString();
    }

    private String getSignature(Class<?> methodOwner, String methodName, Class<?>[] paramTypes) {
        Method matchingMethod = MethodUtil.getMatchingAccessibleMethod(methodOwner, methodName, paramTypes, false);
        StringBuilder signatureBuilder = new StringBuilder();
        signatureBuilder.append('(');
        for(Class<?> paramType : matchingMethod.getParameterTypes()){
            signatureBuilder.append(getJavaType(paramType));
        }
        signatureBuilder.append(')');
        signatureBuilder.append(getJavaType(matchingMethod.getReturnType()));
        return signatureBuilder.toString();
    }

    private void pushFieldToStack(CodeVisitor codeVisitor, int fieldOwnerLocalVarIndex, String fieldName) {
        codeVisitor.visitVarInsn(Constants.ALOAD, 0);
        codeVisitor.visitFieldInsn(Constants.GETFIELD, beanNameWithPackage, fieldName, getJavaType(allFields
                .get(fieldName)));
    }
    
    public String toString() {
        if (StringUtils.isNotBlank(beanName)) {
            return String.format("Bean with name: %s", beanName);
        } 
        return super.toString();
    }
}
