package org.openl.rules.tbasic.compile;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.openl.meta.StringValue;
import org.openl.rules.tbasic.AlgorithmRow;
import org.openl.rules.tbasic.AlgorithmTableParserManager;
import org.openl.rules.tbasic.AlgorithmTreeNode;
import org.openl.source.IOpenSourceCodeModule;
import org.openl.syntax.exception.SyntaxNodeException;
import org.openl.syntax.exception.SyntaxNodeExceptionUtils;
import org.openl.types.IOpenField;
import org.openl.types.java.JavaOpenClass;

public class AlgorithmCompilerTool {

    public static final String FIELD_SEPARATOR = ".";

    private static String extractFieldName(String instruction) {
        // Get the first token after ".", it will be the field name
        return instruction.split(Pattern.quote(FIELD_SEPARATOR))[1];
    }

    private static String extractOperationName(String instruction) {
        // Get the first token before ".", it will be the name of operation
        return instruction.split(Pattern.quote(FIELD_SEPARATOR))[0];
    }

    /**
     * @param candidateNodes
     * @param instruction
     * @return
     * @throws BoundError
     */
    public static AlgorithmTreeNode extractOperationNode(List<AlgorithmTreeNode> candidateNodes, String instruction)
            throws SyntaxNodeException {
        AlgorithmTreeNode executionNode = null;
        String operationName = extractOperationName(instruction);

        for (AlgorithmTreeNode node : candidateNodes) {
            if (operationName.equalsIgnoreCase(node.getAlgorithmRow().getOperation().getValue())) {
                executionNode = node;
            }
        }

        if (executionNode == null) {
            IOpenSourceCodeModule errorSource = candidateNodes.get(0).getAlgorithmRow().getOperation()
                    .asSourceCodeModule();
            throw SyntaxNodeExceptionUtils.createError(String.format("Compilation failure. Can't find %s in operations sequence %s",
                    operationName, candidateNodes), errorSource);
        }
        return executionNode;
    }

    public static Map<String, AlgorithmTreeNode> getAllDeclaredLables(List<AlgorithmTreeNode> nodesToSearch) {
        Map<String, AlgorithmTreeNode> labels = new HashMap<String, AlgorithmTreeNode>();
        for (AlgorithmTreeNode node : nodesToSearch) {
            for (StringValue labelOfNode : node.getLabels()) {
                labels.put(labelOfNode.getValue(), node);
            }
            labels.putAll(getAllDeclaredLables(node.getChildren()));
        }
        return labels;
    }

    /**
     * @param candidateNodes
     * @param instruction
     * @return
     * @throws BoundError
     */
    public static StringValue getCellContent(List<AlgorithmTreeNode> candidateNodes, String instruction)
            throws SyntaxNodeException {
        String fieldName = extractFieldName(instruction);

        AlgorithmTreeNode executionNode = extractOperationNode(candidateNodes, instruction);

        IOpenField codeField = JavaOpenClass.getOpenClass(AlgorithmRow.class).getField(fieldName);

        if (codeField == null) {
            IOpenSourceCodeModule errorSource = candidateNodes.get(0).getAlgorithmRow().getOperation()
                    .asSourceCodeModule();
            throw SyntaxNodeExceptionUtils.createError(String.format("Compilation failure. Can't find %s field", fieldName), errorSource);
        }

        StringValue openLCode = (StringValue) codeField.get(executionNode.getAlgorithmRow(), null);

        return openLCode;
    }

    /**
     *
     * @param nodes
     * @return
     */
    public static AlgorithmTreeNode getLastExecutableOperation(List<AlgorithmTreeNode> nodes) {
        AlgorithmTreeNode lastOperation = nodes.get(nodes.size() - 1);
        if (lastOperation.getSpecification().getKeyword().startsWith("END")) {
            lastOperation = getLastExecutableOperation(nodes.subList(0, nodes.size() - 1));
        } else if (lastOperation.getChildren().size() > 0) {
            lastOperation = getLastExecutableOperation(lastOperation.getChildren());
        }
        return lastOperation;
    }

    /**
     * @param nodesToProcess
     * @param firstNodeIndex
     * @return
     */
    public static int getLinkedNodesGroupSize(List<AlgorithmTreeNode> nodesToProcess, int firstNodeIndex) {
        int linkedNodesGroupSize = 1; // just one operation by default

        AlgorithmTreeNode currentNodeToProcess = nodesToProcess.get(firstNodeIndex);
        String currentNodeKeyword = currentNodeToProcess.getSpecification().getKeyword();

        String[] operationNamesToGroup = AlgorithmTableParserManager.instance().whatOperationsToGroup(
                currentNodeKeyword);

        if (operationNamesToGroup != null) {
            List<String> operationsToGroupWithCurrent = Arrays.asList(operationNamesToGroup);

            for (; linkedNodesGroupSize < nodesToProcess.size() - firstNodeIndex; linkedNodesGroupSize++) {
                AlgorithmTreeNode groupCandidateNode = nodesToProcess.get(firstNodeIndex + linkedNodesGroupSize);
                if (!operationsToGroupWithCurrent.contains(groupCandidateNode.getSpecification().getKeyword())) {
                    break;
                }
            }
        }

        return linkedNodesGroupSize;
    }

    /**
     * @param candidateNodes
     * @param conversionStep
     * @return
     * @throws BoundError
     */
    public static List<AlgorithmTreeNode> getNestedInstructionsBlock(List<AlgorithmTreeNode> candidateNodes,
            String instruction) throws SyntaxNodeException {

        AlgorithmTreeNode executionNode = extractOperationNode(candidateNodes, instruction);

        return executionNode.getChildren();
    }

    /**
     * @param nodesToCompile
     * @param instruction
     * @return
     * @throws BoundError
     */
    public static AlgorithmOperationSource getOperationSource(List<AlgorithmTreeNode> nodesToCompile, String instruction)
            throws SyntaxNodeException {

        AlgorithmTreeNode sourceNode;
        String operationValueName = null;

        // TODO: set more precise source reference
        if (isOperationFieldInstruction(instruction)) {
            sourceNode = extractOperationNode(nodesToCompile, instruction);
            operationValueName = extractFieldName(instruction);
        } else {
            sourceNode = nodesToCompile.get(0);
        }

        return new AlgorithmOperationSource(sourceNode, operationValueName);
    }

    /**
     * @param instruction
     * @return
     */
    public static boolean isOperationFieldInstruction(String instruction) {
        boolean isInstruction = false;

        if (instruction != null) {
            isInstruction = instruction.split(Pattern.quote(FIELD_SEPARATOR)).length == 2;
        }

        return isInstruction;
    }

}
