/*
 * Decompiled with CFR 0.152.
 */
package org.openmuc.framework.lib.amqp;

import com.rabbitmq.client.Channel;
import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConnectionFactory;
import com.rabbitmq.client.ExceptionHandler;
import com.rabbitmq.client.Recoverable;
import com.rabbitmq.client.RecoveryListener;
import com.rabbitmq.client.ShutdownSignalException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeoutException;
import org.openmuc.framework.lib.amqp.AmqpExceptionHandler;
import org.openmuc.framework.lib.amqp.AmqpReader;
import org.openmuc.framework.lib.amqp.AmqpSettings;
import org.openmuc.framework.security.SslManagerInterface;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AmqpConnection {
    private static final Logger logger = LoggerFactory.getLogger(AmqpConnection.class);
    private static final List<String> DECLARED_QUEUES = new ArrayList<String>();
    private final List<RecoveryListener> recoveryListeners = new ArrayList<RecoveryListener>();
    private final List<AmqpReader> readers = new ArrayList<AmqpReader>();
    private final AmqpSettings settings;
    private String exchange;
    private Connection connection;
    private Channel channel;
    private SslManagerInterface sslManager;
    private boolean connected = false;
    private long reconnectTime;

    public AmqpConnection(AmqpSettings settings) throws IOException, TimeoutException {
        this.settings = settings;
        this.reconnectTime = System.currentTimeMillis();
        if (!settings.isSsl()) {
            logger.info("Starting amqp connection without ssl");
            ConnectionFactory factory = this.getConnectionFactoryForSsl(settings);
            try {
                this.connect(settings, factory);
            }
            catch (Exception e) {
                logger.error("Connection could not be created: {}", (Object)e.getMessage());
            }
        }
    }

    private ConnectionFactory getConnectionFactoryForSsl(AmqpSettings settings) {
        ConnectionFactory factory = new ConnectionFactory();
        if (settings.isSsl()) {
            factory.useSslProtocol(this.sslManager.getSslContext());
            factory.enableHostnameVerification();
        }
        factory.setHost(settings.getHost());
        factory.setPort(settings.getPort());
        factory.setVirtualHost(settings.getVirtualHost());
        factory.setUsername(settings.getUsername());
        factory.setPassword(settings.getPassword());
        factory.setExceptionHandler((ExceptionHandler)new AmqpExceptionHandler());
        factory.setRequestedHeartbeat(settings.getConnectionAliveInterval());
        return factory;
    }

    private void connect(AmqpSettings settings, ConnectionFactory factory) throws IOException {
        this.establishConnection(factory);
        if (this.connection == null) {
            logger.warn("Created connection is null, check your config\n{}", (Object)settings);
            return;
        }
        this.connected = true;
        logger.info("Connection established successfully!");
        this.addRecoveryListener(new RecoveryListener(){

            public void handleRecovery(Recoverable recoverable) {
                logger.debug("Connection recovery completed");
                AmqpConnection.this.connected = true;
            }

            public void handleRecoveryStarted(Recoverable recoverable) {
                logger.debug("Connection recovery started");
                AmqpConnection.this.connected = false;
            }
        });
        this.channel = this.connection.createChannel();
        this.exchange = settings.getExchange();
        if (!Objects.equals(this.exchange, "")) {
            this.channel.exchangeDeclare(this.exchange, "topic", true);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Connected to {}:{} on virtualHost {} as user {}", new Object[]{settings.getHost(), settings.getPort(), settings.getVirtualHost(), settings.getUsername()});
        }
    }

    private void establishConnection(ConnectionFactory factory) {
        try {
            this.connection = factory.newConnection();
        }
        catch (Exception e) {
            logger.error("Error at creation of new connection: {}", (Object)e.getMessage());
            this.connection = null;
        }
    }

    private void sslUpdate() {
        logger.warn("SSL configuration changed, reconnecting.");
        this.disconnect();
        ConnectionFactory factory = this.getConnectionFactoryForSsl(this.settings);
        try {
            this.connect(this.settings, factory);
            if (this.connection == null) {
                logger.error("connection after calling ssl update is null");
                return;
            }
            for (RecoveryListener listener : this.recoveryListeners) {
                ((Recoverable)this.connection).addRecoveryListener(listener);
                listener.handleRecovery((Recoverable)this.connection);
            }
            for (AmqpReader reader : this.readers) {
                reader.resubscribe();
            }
        }
        catch (IOException e) {
            logger.error("Reconnection failed. Reason: {}", (Object)e.getMessage());
        }
        logger.warn("Reconnection completed.");
    }

    public void disconnect() {
        if (this.channel == null || this.connection == null) {
            return;
        }
        try {
            this.channel.close();
            this.connection.close();
            if (logger.isTraceEnabled()) {
                logger.trace("Successfully disconnected");
            }
        }
        catch (ShutdownSignalException | IOException | TimeoutException e) {
            logger.error("failed to close connection: {}", (Object)e.getMessage());
        }
    }

    public void declareQueue(String queue) throws IOException {
        if (!DECLARED_QUEUES.contains(queue)) {
            try {
                this.channel.queueDeclarePassive(queue);
                this.channel.queueBind(queue, this.exchange, queue);
                DECLARED_QUEUES.add(queue);
                if (logger.isTraceEnabled()) {
                    logger.trace("Queue {} declared", (Object)queue);
                }
            }
            catch (Exception e) {
                logger.debug("Channel {} not found, start to create it...", (Object)queue);
                this.initDeclare(queue);
            }
        }
    }

    void addRecoveryListener(RecoveryListener listener) {
        this.recoveryListeners.add(listener);
        if (this.connection == null) {
            return;
        }
        ((Recoverable)this.connection).addRecoveryListener(listener);
    }

    void addReader(AmqpReader reader) {
        this.readers.add(reader);
    }

    private void initDeclare(String queue) throws IOException {
        this.ensureConnected(false);
        if (this.connection == null) {
            logger.error("declaring queue stopped, because connection to broker is null");
            return;
        }
        try {
            this.channel = this.connection.createChannel();
        }
        catch (Exception e) {
            logger.error("Queue {} could not be declared.", (Object)queue);
            return;
        }
        this.channel.exchangeDeclare(this.exchange, "topic", true);
        this.channel.queueDeclare(queue, true, false, false, null);
    }

    public void ensureConnected(boolean forceReconnect) {
        if (forceReconnect || this.connection == null) {
            if (this.reconnectTime >= System.currentTimeMillis()) {
                logger.error("Not connected, trying to reconnect in {}s", (Object)((this.reconnectTime - System.currentTimeMillis()) / 1000L), (Object)this.settings.getReconnectionInterval());
                return;
            }
            logger.error("Not connected, trying to reconnect...");
            this.reconnectTime = System.currentTimeMillis() + (long)(this.settings.getReconnectionInterval() * 1000);
            this.connected = false;
            ConnectionFactory factory = this.getConnectionFactoryForSsl(this.settings);
            try {
                this.connect(this.settings, factory);
            }
            catch (IOException e) {
                logger.error("Reconnect failed: {}", (Object)e.getMessage());
                return;
            }
            if (this.connection == null) {
                logger.error("Reconnect failed.");
            } else {
                logger.info("Successfully reconnected.");
                for (RecoveryListener listener : this.recoveryListeners) {
                    ((Recoverable)this.connection).addRecoveryListener(listener);
                    listener.handleRecovery((Recoverable)this.connection);
                }
            }
        }
    }

    public String getExchange() {
        return this.exchange;
    }

    Channel getRabbitMqChannel() {
        return this.channel;
    }

    AmqpSettings getSettings() {
        return this.settings;
    }

    public void setSslManager(SslManagerInterface instance) {
        if (!this.settings.isSsl()) {
            return;
        }
        this.sslManager = instance;
        this.sslManager.listenForConfigChange(this::sslUpdate);
        ConnectionFactory factory = this.getConnectionFactoryForSsl(this.settings);
        if (this.sslManager.isLoaded()) {
            try {
                this.connect(this.settings, factory);
            }
            catch (Exception e) {
                logger.error("Connection with SSL couldn't be created");
            }
        }
    }

    public boolean isConnected() {
        return this.connected;
    }
}

