/*
 * Decompiled with CFR 0.152.
 */
package org.openmuc.framework.lib.filePersistence;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FilePersistence {
    private static final Logger logger = LoggerFactory.getLogger(FilePersistence.class);
    private final Path DIRECTORY;
    private int maxFileCount;
    private final long MAX_FILE_SIZE_BYTES;
    private final Map<String, Integer> nextFile = new HashMap<String, Integer>();
    private final Map<String, Long> readBytes = new HashMap<String, Long>();
    private static final List<String> BUFFERS = new ArrayList<String>();
    public static final String DEFAULT_FILENAME = "buffer.0.log";
    public static final String DEFAULT_FILE_PREFIX = "buffer";
    public static final String DEFAULT_FILE_SUFFIX = "log";

    public FilePersistence(String directory, int maxFileCount, long maxFileSizeKb) {
        this.DIRECTORY = FileSystems.getDefault().getPath(directory, new String[0]);
        this.MAX_FILE_SIZE_BYTES = maxFileSizeKb * 1024L;
        this.setMaxFileCount(maxFileCount);
        this.createDirectory();
    }

    private void setMaxFileCount(int maxFileCount) {
        this.maxFileCount = maxFileCount;
        if (this.maxFileCount <= 0) {
            throw new IllegalArgumentException("maxFileSize is 0");
        }
    }

    private void createDirectory() {
        if (!this.DIRECTORY.toFile().exists() && !this.DIRECTORY.toFile().mkdirs()) {
            logger.error("The directory {} could not be created", (Object)this.DIRECTORY);
        }
    }

    public void writeBufferToFile(String buffer, byte[] payload) throws IOException {
        this.checkPayLoadSize(payload.length);
        this.registerBuffer(buffer);
        Path filePath = Paths.get(this.DIRECTORY.toString(), buffer, DEFAULT_FILENAME);
        File file = this.createFileIfNotExist(filePath);
        if (this.isFileFull(file.length(), payload.length)) {
            this.handleFullFile(buffer, payload, file);
        } else {
            this.appendToFile(file, payload);
        }
    }

    private void registerBuffer(String buffer) throws IOException {
        if (!BUFFERS.contains(buffer)) {
            BUFFERS.add(buffer);
            this.writeBufferList();
        }
    }

    private void removeBufferIfEmpty(String buffer) throws IOException {
        boolean buffersChanged;
        if (!this.fileExistsFor(buffer) && (buffersChanged = BUFFERS.remove(buffer))) {
            this.writeBufferList();
        }
    }

    private void writeBufferList() throws IOException {
        File buffers = Paths.get(this.DIRECTORY.toString(), "buffer_list").toFile();
        FileWriter writer = new FileWriter(buffers);
        for (String registeredBuffer : BUFFERS) {
            writer.write(registeredBuffer + '\n');
        }
        writer.close();
    }

    public String[] getBuffers() {
        Path buffers;
        if (BUFFERS.isEmpty() && (buffers = Paths.get(this.DIRECTORY.toString(), "buffer_list")).toFile().exists()) {
            try {
                BUFFERS.addAll(Files.readAllLines(buffers));
            }
            catch (IOException e) {
                logger.error("Could not read buffer_list. Message: {}", (Object)e.getMessage());
            }
        }
        return BUFFERS.toArray(new String[0]);
    }

    private File createFileIfNotExist(Path filePath) throws IOException {
        File file = filePath.toFile();
        if (!file.exists()) {
            logger.info("create new file: {}", (Object)file.getAbsolutePath());
            Path storagePath = Paths.get(file.getAbsolutePath(), new String[0]);
            Files.createDirectories(storagePath.getParent(), new FileAttribute[0]);
            Files.createFile(storagePath, new FileAttribute[0]);
        }
        return file;
    }

    private void appendToFile(File file, byte[] payload) throws IOException {
        FileOutputStream fileStream = new FileOutputStream(file, true);
        fileStream.write(payload);
        fileStream.write("\n".getBytes());
        fileStream.close();
    }

    private void handleFullFile(String filePath, byte[] payload, File file) throws IOException {
        if (this.maxFileCount > 1) {
            this.handleMultipleFiles(filePath, payload, file);
        } else {
            this.handleSingleFile(filePath, payload, file);
        }
    }

    private void handleSingleFile(String filePath, byte[] payload, File file) {
        throw new UnsupportedOperationException("right now only maxFileCount >= 2 supported");
    }

    private void handleMultipleFiles(String buffer, byte[] payload, File file) throws IOException {
        int nextFile = this.nextFile.getOrDefault(buffer, 1);
        String newFileName = "buffer." + nextFile + '.' + DEFAULT_FILE_SUFFIX;
        if (++nextFile == this.maxFileCount) {
            nextFile = 1;
        }
        this.nextFile.put(buffer, nextFile);
        Path path = Paths.get(this.DIRECTORY.toString(), buffer, DEFAULT_FILENAME);
        Path newPath = Paths.get(this.DIRECTORY.toString(), buffer, newFileName);
        Files.move(path, newPath, StandardCopyOption.REPLACE_EXISTING);
        logger.info("move file from: {} to {}", (Object)path, (Object)newPath);
        Files.createFile(path, new FileAttribute[0]);
        this.appendToFile(path.toFile(), payload);
    }

    private boolean isFileFull(long fileLength, int payloadLength) {
        return fileLength + (long)payloadLength + 1L > this.MAX_FILE_SIZE_BYTES;
    }

    private void checkPayLoadSize(int payloadLength) throws IOException {
        if ((long)payloadLength >= this.MAX_FILE_SIZE_BYTES) {
            throw new IOException("Payload is bigger than maxFileSize. Current maxFileSize is " + this.MAX_FILE_SIZE_BYTES / 1024L + "kB");
        }
    }

    public boolean fileExistsFor(String buffer) {
        String fileName = DEFAULT_FILENAME;
        return Paths.get(this.DIRECTORY.toString(), buffer, fileName).toFile().exists();
    }

    public byte[] getMessage(String buffer) {
        Path filePath = this.getOldestFilePath(buffer);
        Long position = this.getFilePosition(filePath.toString());
        String message = "";
        try {
            message = this.readLine(buffer, filePath.toFile(), position);
        }
        catch (IOException e) {
            logger.error("An error occurred while reading the buffer {}. Error message: {}", (Object)buffer, (Object)e.getMessage());
        }
        return message.getBytes();
    }

    private String readLine(String buffer, File file, Long position) throws IOException {
        FileInputStream fileStream = new FileInputStream(file);
        StringBuilder lineBuilder = new StringBuilder();
        fileStream.skip(position);
        int nextChar = fileStream.read();
        Long l = position;
        Long l2 = position = Long.valueOf(position + 1L);
        while (nextChar != -1 && nextChar != 10) {
            lineBuilder.appendCodePoint(nextChar);
            nextChar = fileStream.read();
            l = position;
            l2 = position = Long.valueOf(position + 1L);
        }
        fileStream.close();
        this.setFilePosition(file.toString(), position);
        this.deleteIfEmpty(file, position);
        this.removeBufferIfEmpty(buffer);
        return lineBuilder.toString();
    }

    private void deleteIfEmpty(File file, Long position) throws IOException {
        FileInputStream fileStream = new FileInputStream(file);
        fileStream.skip(position);
        int nextChar = fileStream.read();
        fileStream.close();
        if (nextChar == -1) {
            boolean deleted = file.delete();
            if (!deleted) {
                throw new IOException("Empty file could not be deleted!");
            }
            this.setFilePosition(file.toString(), 0L);
        }
    }

    private void setFilePosition(String filePathString, Long position) {
        this.readBytes.put(filePathString, position);
    }

    private Long getFilePosition(String filePathString) {
        Long position = this.readBytes.get(filePathString);
        if (position == null) {
            position = 0L;
        }
        return position;
    }

    private Path getOldestFilePath(String buffer) {
        Path directoryPath = Paths.get(this.DIRECTORY.toString(), buffer);
        String[] bufferFiles = directoryPath.toFile().list((file, s) -> s.endsWith(".log"));
        String oldestFile = DEFAULT_FILENAME;
        if (bufferFiles.length > 1) {
            oldestFile = this.findOldestFile(buffer);
        }
        return Paths.get(directoryPath.toString(), oldestFile);
    }

    private String findOldestFile(String buffer) {
        String oldestFile = DEFAULT_FILENAME;
        int nextFile = this.nextFile.getOrDefault(buffer, 1);
        for (int i = 0; i < this.maxFileCount; ++i) {
            String fileName = "buffer." + nextFile + '.' + DEFAULT_FILE_SUFFIX;
            if (Paths.get(this.DIRECTORY.toString(), buffer, fileName).toFile().exists()) {
                oldestFile = fileName;
                break;
            }
            if (++nextFile != this.maxFileCount) continue;
            nextFile = 1;
        }
        return oldestFile;
    }

    public void restructure() throws IOException {
        for (String buffer : this.getBuffers()) {
            Path bufferPath = this.getOldestFilePath(buffer);
            Long position = this.getFilePosition(bufferPath.toString());
            if (position.equals(0L)) continue;
            Path temp = bufferPath.getParent();
            temp = Paths.get(temp.toString(), "temp");
            try {
                Files.move(bufferPath, temp, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (DirectoryNotEmptyException e) {
                logger.error(bufferPath.toString() + " -> " + temp.toString());
            }
            Files.createFile(bufferPath, new FileAttribute[0]);
            FileInputStream inputStream = new FileInputStream(temp.toFile());
            inputStream.skip(position);
            FileOutputStream outputStream = new FileOutputStream(bufferPath.toFile(), true);
            int nextChar = inputStream.read();
            while (nextChar != -1) {
                outputStream.write(nextChar);
                nextChar = inputStream.read();
            }
            inputStream.close();
            outputStream.close();
            temp.toFile().delete();
            this.setFilePosition(bufferPath.toString(), 0L);
        }
    }
}

