/*
 * Copyright (C) 2021 The ORT Project Authors (see <https://github.com/oss-review-toolkit/ort/blob/main/NOTICE>)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * License-Filename: LICENSE
 */

package org.ossreviewtoolkit.model.config

import org.ossreviewtoolkit.model.vulnerabilities.Vulnerability

/**
 * Defines the resolution of an [Vulnerability]. This can be used to silence false positives, or vulnerabilities that
 * have been identified as not being relevant.
 */
data class VulnerabilityResolution(
    /**
     * A regular expression to match the id, e.g. a CVE, of the vulnerability to resolve. Will be converted to a [Regex]
     * using [RegexOption.DOT_MATCHES_ALL].
     */
    val id: String,

    /**
     * The reason why the vulnerability is resolved.
     */
    val reason: VulnerabilityResolutionReason,

    /**
     * A comment to further explain why the [reason] is applicable here.
     */
    val comment: String
) {
    private val regex = Regex(id, RegexOption.DOT_MATCHES_ALL)

    /**
     * True if [id] matches the id of [vulnerability].
     */
    fun matches(vulnerability: Vulnerability) = regex.matches(vulnerability.id)
}
