/*
 * Copyright (C) 2021 The ORT Project Authors (see <https://github.com/oss-review-toolkit/ort/blob/main/NOTICE>)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * License-Filename: LICENSE
 */

package org.ossreviewtoolkit.model.config

import org.ossreviewtoolkit.model.vulnerabilities.Vulnerability

/**
 * Possible reasons for resolving an [Vulnerability] using a [VulnerabilityResolution].
 */
enum class VulnerabilityResolutionReason {
    /**
     * No remediation is available for this vulnerability, e.g., because it requires a change to be made
     * by a third party that is not responsive.
     */
    CANT_FIX_VULNERABILITY,

    /**
     * The code in which the vulnerability was found is neither invoked in the project's code nor indirectly
     * via another open source component.
     */
    INEFFECTIVE_VULNERABILITY,

    /**
     * The vulnerability is irrelevant due to a tooling or database mismatch, e.g., the package version used
     * does not match the version for which the vulnerability provider has reported a vulnerability.
     */
    INVALID_MATCH_VULNERABILITY,

    /**
     * The vulnerability is valid but has been mitigated, e.g., measures have been taken to ensure
     * this vulnerability can not be exploited.
     */
    MITIGATED_VULNERABILITY,

    /**
     * This vulnerability will never be fixed, e.g., because the package which is affected is orphaned,
     * declared end-of-life, or otherwise deprecated.
     */
    WILL_NOT_FIX_VULNERABILITY,

    /**
     * The vulnerability is valid but a temporary workaround has been put in place to avoid exposure
     * to the vulnerability.
     */
    WORKAROUND_FOR_VULNERABILITY
}
