/*
 * Copyright (C) 2017 The ORT Project Authors (see <https://github.com/oss-review-toolkit/ort/blob/main/NOTICE>)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * License-Filename: LICENSE
 */

package org.ossreviewtoolkit.model.utils

import org.ossreviewtoolkit.model.Issue
import org.ossreviewtoolkit.model.RuleViolation
import org.ossreviewtoolkit.model.config.IssueResolution
import org.ossreviewtoolkit.model.config.RuleViolationResolution
import org.ossreviewtoolkit.model.config.VulnerabilityResolution
import org.ossreviewtoolkit.model.vulnerabilities.Vulnerability

/**
 * An interface to provide resolutions for [Issue]s, [RuleViolation]s and [Vulnerability]s .
 */
interface ResolutionProvider {
    /**
     * Get all issue resolutions that match [issue].
     */
    fun getResolutionsFor(issue: Issue): List<IssueResolution>

    /**
     * Get all rule violation resolutions that match [violation].
     */
    fun getResolutionsFor(violation: RuleViolation): List<RuleViolationResolution>

    /**
     * Get all vulnerability resolutions that match [vulnerability].
     */
    fun getResolutionsFor(vulnerability: Vulnerability): List<VulnerabilityResolution>

    /**
     * Return true if there is at least one issue resolution that matches [issue].
     */
    fun isResolved(issue: Issue): Boolean = getResolutionsFor(issue).isNotEmpty()

    /**
     * Return true if there is at least one rule violation resolution that matches [violation].
     */
    fun isResolved(violation: RuleViolation): Boolean = getResolutionsFor(violation).isNotEmpty()

    /**
     * Return true if there is at least one vulnerability resolution that matches [vulnerability].
     */
    fun isResolved(vulnerability: Vulnerability): Boolean = getResolutionsFor(vulnerability).isNotEmpty()
}
