/*
 * Copyright (C) 2021 The ORT Project Authors (see <https://github.com/oss-review-toolkit/ort/blob/main/NOTICE>)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * License-Filename: LICENSE
 */

package org.ossreviewtoolkit.model.vulnerabilities

import com.fasterxml.jackson.annotation.JsonIgnoreProperties

import java.net.URI

/**
 * A data class representing detailed information about a vulnerability obtained from a specific source.
 *
 * A single vulnerability can be listed by multiple sources using different scoring systems to denote its severity.
 * So when ORT queries different providers for vulnerability information it may well find multiple records for a single
 * vulnerability, which could even contain contradicting information. To model this, a [Vulnerability] is associated
 * with a list of references; each reference points to the source of the information and has some detailed information
 * provided by this source.
 */
@JsonIgnoreProperties(value = ["severity_rating"], allowGetters = true)
data class VulnerabilityReference(
    /**
     * The URI pointing to details of this vulnerability. This can also be used to derive the source of this
     * information.
     */
    val url: URI,

    /**
     * The name of the scoring system to express the severity of this vulnerability if available.
     */
    val scoringSystem: String?,

    /**
     * The severity assigned to the vulnerability by this reference. Note that this is a plain string, whose meaning
     * depends on the concrete scoring system. It could be a number, but also a constant like _LOW_ or _HIGH_. A
     * *null* value is possible as well, meaning that this reference does not contain any information about the
     * severity.
     */
    val severity: String?
) {
    companion object {
        private val CVSS3_SEVERITIES = Cvss3Rating.entries.map { it.name }

        /**
         * Return a human-readable string that is determined based on the given [scoringSystem] and [severity].
         */
        fun getSeverityString(scoringSystem: String?, severity: String?): String =
            when (scoringSystem?.uppercase()) {
                in Cvss2Rating.NAMES -> severity?.toFloatOrNull()?.let { Cvss2Rating.fromScore(it) }?.toString()
                in Cvss3Rating.NAMES -> severity?.toFloatOrNull()?.let { Cvss3Rating.fromScore(it) }?.toString()
                in Cvss4Rating.NAMES -> severity?.toFloatOrNull()?.let { Cvss4Rating.fromScore(it) }?.toString()
                else -> severity?.uppercase()?.takeIf { it in CVSS3_SEVERITIES }
            } ?: "UNKNOWN"
    }

    /**
     * Return a human-readable severity rating string.
     */
    val severityRating: String by lazy { getSeverityString(scoringSystem, severity) }
}
