/**
 * EasyBeans
 * Copyright (C) 2007-2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CmiLifeCycleCallback.java 2900 2008-04-22 14:20:02Z alitokmen $
 * --------------------------------------------------------------------------
 */

package org.ow2.easybeans.component.cmi;

import java.util.Set;

import org.ow2.cmi.annotation.Cluster;
import org.ow2.cmi.info.CMIInfoExtractor;
import org.ow2.cmi.info.CMIInfoExtractorException;
import org.ow2.cmi.info.CMIInfoRepository;
import org.ow2.cmi.info.ClusteredObjectInfo;
import org.ow2.easybeans.api.EZBContainerCallbackInfo;
import org.ow2.easybeans.api.Factory;
import org.ow2.easybeans.api.LifeCycleCallbackException;
import org.ow2.easybeans.api.binding.EZBRef;
import org.ow2.easybeans.container.EmptyLifeCycleCallBack;
import org.ow2.easybeans.container.session.SessionFactory;
import org.ow2.easybeans.container.session.stateful.StatefulSessionFactory;
import org.ow2.easybeans.proxy.reference.RemoteCallRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * Provide a callback for CMI.
 * @author WEI Zhouyue & ZHU Ning
 *         Contributors:
 *             S. Ali Tokmen (JNDI naming strategy)
 */
public class CmiLifeCycleCallback extends EmptyLifeCycleCallBack {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(CmiLifeCycleCallback.class);

    /**
     * Called before binding a reference into the registry.
     * Extract the informations on clustering and register it in the repository of CMI.
     * @param info some information on the container which is running.
     * @param reference a reference on the bean that will be bound
     * @throws LifeCycleCallbackException if the invocation of the callback failed
     */
    @Override
    public void beforeBind(final EZBContainerCallbackInfo info, final EZBRef reference)
    throws LifeCycleCallbackException {
        // CMI only works on remote interfaces
        if (reference instanceof RemoteCallRef) {
            Factory<?, ?> factory = reference.getFactory();
            if (factory instanceof SessionFactory) {
                Class<?> beanClass = ((SessionFactory<?>) factory).getBeanClass();
                if (beanClass.isAnnotationPresent(Cluster.class)) {
                    Class<?> itfClass = null;
                    String itfClassname = reference.getItfClassName();
                    logger.info("The bean with the jndi name {0} and the interface name {1} is clustered.",
                            reference.getJNDIName(), itfClassname);
                    for (Class<?> klass : beanClass.getInterfaces()) {
                        if (klass.getName().equals(itfClassname)) {
                            itfClass = klass;
                        }
                    }
                    if (itfClass == null) {
                        logger.error("Cannot find the interface for name {0}", itfClassname);
                        throw new LifeCycleCallbackException("Cannot find the interface for name " + itfClassname);
                    }
                    Set<String> applicationExceptionNames = factory.getBeanInfo().getApplicationExceptions().keySet();
                    // Extract the informations on clustering
                    ClusteredObjectInfo infos;
                    try {
                        infos =
                            CMIInfoExtractor.extractClusteringInfoFromAnnotatedPOJO(
                                    reference.getJNDIName(), itfClass, beanClass,
                                    factory instanceof StatefulSessionFactory, false, applicationExceptionNames);
                    } catch (CMIInfoExtractorException e) {
                        logger.error("Cannot extract infos for the class with name {0}", beanClass.getName(), e);
                        throw new LifeCycleCallbackException(
                                "Cannot extract infos for the class with name " + beanClass.getName(), e);
                    }
                    // Register the informations in the repository of CMI
                    CMIInfoRepository.addClusteredObjectInfo(reference.getJNDIName(), infos);
                }
            }
        }
    }

}
