/**
 * EasyBeans
 * Copyright (C) 2008 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ContainerJNDIResolver.java 4038 2008-10-01 13:35:21Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.easybeans.resolver;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ow2.easybeans.resolver.api.EZBApplicationJNDIResolver;
import org.ow2.easybeans.resolver.api.EZBContainerJNDIResolver;
import org.ow2.easybeans.resolver.api.EZBJNDIData;
import org.ow2.util.archive.api.ArchiveException;
import org.ow2.util.archive.api.IArchive;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

/**
 * JNDI resolver for a container. It can answer to any request about beans of a given container.
 * @author Florent Benoit
 */
public class ContainerJNDIResolver extends CommonJNDIResolver implements EZBContainerJNDIResolver {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(ContainerJNDIResolver.class);

    /**
     * URL of this container resolver.
     */
    private URL containerURL = null;

    /**
     * Link to an Application JNDI Resolver.
     */
    private EZBApplicationJNDIResolver applicationJNDIResolver = null;

    /**
     * Map beween interface name and a map between bean name and the JNDI data.
     */
    private Map<String, Map<String, EZBJNDIData>> interfacesMap;


    /**
     * Default constructor.
     * @param archive the archive of the given container.
     */
    public ContainerJNDIResolver(final IArchive archive) {
        try {
            this.containerURL = archive.getURL();
        } catch (ArchiveException e) {
            throw new IllegalArgumentException("Cannot get URL from archive '" + archive + "'.");
        }
        this.interfacesMap = new HashMap<String, Map<String, EZBJNDIData>>();
    }

    /**
     * @return URL used by this resolver.
     */
    public URL getURL() {
        return this.containerURL;
    }

    /**
     * Allows to find EJB JNDI name.
     * @return a list of matching JNDI objects for the given interface and bean
     *         name.
     * @param interfaceName the name of the interface that EJBs are
     *        implementing.
     */
    public List<EZBJNDIData> getNames(final String interfaceName) {
        // Return a list with all the beans available for this interface
        return getNames(interfaceName, null);
    }

    /**
     * @return the application JNDI Resolver.
     */
    public EZBApplicationJNDIResolver getApplicationJNDIResolver() {
        return this.applicationJNDIResolver;
    }

    /**
     * Sets the application resolver.
     * @param applicationJNDIResolver the application resolver
     */
    public void setApplicationJNDIResolver(final EZBApplicationJNDIResolver applicationJNDIResolver) {
        this.applicationJNDIResolver = applicationJNDIResolver;
    }

    /**
     * Allows to find EJB JNDI name.
     * @return a list of matching JNDI objects for the given interface and bean
     *         name.
     * @param interfaceName the name of the interface that EJBs are
     *        implementing.
     * @param beanName the name of the bean on which we need to find JNDI name.
     */
    public List<EZBJNDIData> getNames(final String interfaceName, final String beanName) {
        // Ask also EAR if present
        return getNames(interfaceName, beanName, true);
    }

    /**
     * Allows to find EJB JNDI name.
     * @return a list of matching JNDI objects for the given interface and bean
     *         name.
     * @param interfaceName the name of the interface that EJBs are
     *        implementing.
     * @param beanName the name of the bean on which we need to find JNDI name.
     * @param askParent allow to disallow request to parent.
     */
    public List<EZBJNDIData> getNames(final String interfaceName, final String beanName, final boolean askParent) {

        // Get available beans for the given interface name
        Map<String, EZBJNDIData> beansMap = this.interfacesMap.get(interfaceName);

        // No beans for the given interface
        if (beansMap == null) {
            // Not found, check in the other containers of the EAR (if any)
            return getJNDINameInEAR(interfaceName, beanName, askParent);
        }

        // No bean name, use all values
        if (beanName == null || "".equals(beanName)) {
            Collection<EZBJNDIData> beanValues = beansMap.values();

            // Not found, check in the other containers of the EAR
            if (beanValues == null || beanValues.isEmpty()) {
                // Not found, check in the other containers of the EAR (if any)
                return getJNDINameInEAR(interfaceName, beanName, askParent);
            }

            // Return a list with all the beans available for this interface
            return new ArrayList<EZBJNDIData>(beanValues);
        }

        // Is that the bean name exists and there is an ejb-link in this name ?
        String newBeanname = beanName;
        if (beanName.indexOf("#") > 0) {
            // extract only the bean name
            newBeanname = beanName.split("#")[1];
            logger.debug("EJB-LINK not fully supported for interface '" + interfaceName
                    + "', and bean name '" + beanName + "' in container '" + this.containerURL + "'.");
        }

        // Build list for the return value
        List<EZBJNDIData> beanDataList = new ArrayList<EZBJNDIData>();

        // Bean name is here, use it
        EZBJNDIData beanData = beansMap.get(newBeanname);

        // Found a value, add it
        if (beanData == null) {
            // Not found, check in the other containers of the EAR (if any)
            return getJNDINameInEAR(interfaceName, newBeanname, askParent);
        }
        // Add the local instance found
        beanDataList.add(beanData);

        // Return value
        return beanDataList;
    }

    /**
     * Allows to find EJB JNDI name in the EAR if the container is packaged
     * within an EAR.
     * @return a list of matching JNDI objects for the given interface and bean
     *         name.
     * @param interfaceName the name of the interface that EJBs are
     *        implementing.
     * @param beanName the name of the bean on which we need to find JNDI name.
     * @param enabled allow to disallow request to parent.
     */
    protected List<EZBJNDIData> getJNDINameInEAR(final String interfaceName, final String beanName, final boolean enabled) {
        // If container is not inside an EAR or that recursion is turned off
        // stop here
        if (!enabled || this.applicationJNDIResolver == null) {
            return Collections.emptyList();
        }

        // Ask the resolver of the EAR
        return this.applicationJNDIResolver.getNames(interfaceName, beanName);
    }


    /**
     * Adds a new JNDI name for the given interface name / bean name.
     * @param interfaceName the given interface
     * @param jndiData data for the JNDI Name entry
     */
    public void addJNDIName(final String interfaceName, final EZBJNDIData jndiData) {
        // Get beans map
        Map<String, EZBJNDIData> beansMap = this.interfacesMap.get(interfaceName);

        // Null ? create a new one
        if (beansMap == null) {
            beansMap = new HashMap<String, EZBJNDIData>();
            this.interfacesMap.put(interfaceName, beansMap);
        }

        String beanName = jndiData.getBeanName();

        // Existing info about the bean ?
        EZBJNDIData existingData = beansMap.get(beanName);
        if (existingData != null) {
            logger.warn("Data already set for '" + jndiData + "' for the container URL '" + this.containerURL + "'.");
        }

        // Put info
        beansMap.put(beanName, jndiData);

    }
}
