/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2018 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.logger;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Deque;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Formatter;
import java.util.logging.LogRecord;

import com.ebmwebsourcing.easycommons.lang.reflect.ReflectionHelper;

/**
 * <p>This formatter is aimed at using first {@link LogRecord} parameter as
 * "just in time evaluated" extra log information.</p>
 * <p>Note that localization is still possible (parameters are shifted left if
 * necessary).</p>
 * <p>
 * In other words :
 * </p>
 * <ul>
 * <li>if first parameter is an instance of {@link LogData}, this data will be
 * appended in log message,</li>
 * <li>otherwise, first parameter is kept for localization duty, as usual.</li>
 * </ul>
 * 
 * @author mjambert
 * 
 */
public class LogDataFormatter extends Formatter {

    private static final String LOGDATA_CANONICAL_CLASSNAME = "com.ebmwebsourcing.easycommons.logger.LogData";

    public static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss,SSS 'GMT'Z",
            Locale.getDefault());

    private final Deque<LogDataAppender> dataAppenders;

    private String startingDelimiter = " {{";

    private String endingDelimiter = "}}";

    private String prefix = null;
    
    public void setExtraParametersDelimiter(String startingDelimiter, String endingDelimiter) {
        this.startingDelimiter = startingDelimiter;
        this.endingDelimiter = endingDelimiter;
    }

    public LogDataFormatter() {
        dataAppenders = new LinkedList<LogDataAppender>();
    }

    public final void addLogDataAppender(LogDataAppender logDataAppender) {
        assert logDataAppender != null;
        dataAppenders.add(logDataAppender);
    }

    final LogDataAppender[] getLogDataAppenders() {
        return dataAppenders.toArray(new LogDataAppender[dataAppenders.size()]);
    }
    
    public final void setPrefix(String prefix) {
    	this.prefix = prefix;
    }

    protected void formatExtraParameters(StringBuilder outBuffer, Map<String, Object> extraParameters) {
        Iterator<String> extraParametersKeySetIterator = extraParameters.keySet().iterator();
        if (extraParametersKeySetIterator.hasNext() && !dataAppenders.isEmpty()) {
            outBuffer.append(this.startingDelimiter);
            while (extraParametersKeySetIterator.hasNext()) {
                String extraParameterKey = extraParametersKeySetIterator.next();
                Iterator<LogDataAppender> itAppender = dataAppenders.descendingIterator();
                while (itAppender.hasNext()) {
                    LogDataAppender appender = (LogDataAppender) itAppender.next();
                    if (!appender.canAppend(extraParameters, extraParameterKey)) {
                        continue;
                    }
                    appender.append(outBuffer, extraParameters, extraParameterKey);
                    if (extraParametersKeySetIterator.hasNext()) {
                        outBuffer.append(", ");
                    }
                    break;
                }
            }
            outBuffer.append(this.endingDelimiter);
        }
    }

    private boolean isThereLogDataParameter(Object[] logParameters) {
        return (logParameters != null)
                && (logParameters.length != 0)
                && logParameters[0] != null
                && ReflectionHelper.isOrInheritedFrom(logParameters[0].getClass(),
                        LOGDATA_CANONICAL_CLASSNAME);
    }

    @SuppressWarnings("unchecked")
    protected void populateExtraParameters(Map<String, Object> extraParameters,
            LogRecord logRecord) {
        Object[] logParameters = logRecord.getParameters();
        if (isThereLogDataParameter(logParameters)) {
            extraParameters.putAll((Map<String, Object>) logParameters[0]);
        }
    }

    protected StringBuilder buildPattern(LogRecord record) {
        StringBuilder sb = new StringBuilder();
        String formattedLog;
        if(this.prefix == null) {
        	formattedLog = String.format("%s %s [%s] : %s", DATE_FORMAT.format(record.getMillis()),
                record.getLevel(), record.getLoggerName(), record.getMessage());
        } else {
        	formattedLog = String.format("%s %s %s [%s] : %s", prefix, DATE_FORMAT.format(record.getMillis()),
                    record.getLevel(), record.getLoggerName(), record.getMessage());
        }
        sb.append(formattedLog);
        return sb;
    }

    @Override
    public String format(LogRecord record) {
        Map<String, Object> extraParameters = new LinkedHashMap<String, Object>();
        populateExtraParameters(extraParameters, record);

        StringBuilder sb = buildPattern(record);
        formatExtraParameters(sb, extraParameters);
        sb.append("\n");

        if (record.getThrown() != null) {
            final StringWriter sw = new StringWriter();
            record.getThrown().printStackTrace(new PrintWriter(sw));
            sw.flush();
            sb.append(sw.toString());
        }
        return sb.toString();
    }

    @Override
    public synchronized String formatMessage(LogRecord record) {
        Object[] extraParameters = record.getParameters();
        String result = null;
        if (isThereLogDataParameter(extraParameters)) {
            Object[] extraParametersWithoutLogData = Arrays.copyOfRange(extraParameters, 1,
                    extraParameters.length);
            record.setParameters(extraParametersWithoutLogData);
            result = super.formatMessage(record);
            record.setParameters(extraParameters);
        } else {
            result = super.formatMessage(record);
        }
        return result;
    }
}
