/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2018 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.logger;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Date;

import org.junit.Test;

public class BasicLogDataAppenderTest extends AbstractLogDataAppenderTest {

    private static final String TEST_PREFIX = "TESTPREFIX : ";

    
    @Test
    public void testCanAppendNullDataValue() {
        LogData logData = newTestLogData("id", null);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendStringDataValue() {
        LogData logData = newTestLogData("id", "kltsjlfkgjskl");
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendShortDataValue() {
        LogData logData = newTestLogData("id", (short) 14);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendIntegerDataValue() {
        LogData logData = newTestLogData("id", 145);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendLongDataValue() {
        LogData logData = newTestLogData("id", 145L);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendDoubleDataValue() {
        LogData logData = newTestLogData("id", 5.4D);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendFloatDataValue() {
        LogData logData = newTestLogData("id", 5.4F);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendByteDataValue() {
        LogData logData = newTestLogData("id", (byte) 10);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendBigIntegerDataValue() {
        LogData logData = newTestLogData("id", new BigInteger("145"));
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendBigDecimalDataValue() {
        LogData logData = newTestLogData("id", new BigDecimal(145.45));
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendCharDataValue() {
        LogData logData = newTestLogData("id", 'c');
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendBooleanDataValue() {
        LogData logData = newTestLogData("id", true);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    private static enum TestEnum {
        ENUM_VAL1, ENUM_VAL2
    }

    @Test
    public void testCanAppendEnumDataValue() {
        LogData logData = newTestLogData("id", TestEnum.ENUM_VAL1);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testCanAppendObjectDataValue() {
        LogData logData = newTestLogData("id", new Date());
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }
    
    
    @Test
    public void testCanAppendStringArrayDataValue() {
        LogData logData = newTestLogData("id", new String[] { "s1", "s2", "s3" });
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }
    
    @Test
    public void testCanAppendIntArrayDataValue() {
        LogData logData = newTestLogData("id", new int[] { 0, 1, 2 });
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }
    
    @Test
    public void testCanAppendIntegerArrayDataValue() {
        LogData logData = newTestLogData("id", new Integer[] { 0, 1, 2 });
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }
    
    @Test
    public void testCanAppendObjectArrayDataValue() {
        LogData logData = newTestLogData("id", new Object[] { 0, 1, 2 });
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }
    
    @Test
    public void testCanAppendObjectArrayWithNullDataValue() {
        LogData logData = newTestLogData("id", new Object[] { 0, null, 2 });
        BasicLogDataAppender appender = new BasicLogDataAppender();
        assertTrue(appender.canAppend(logData, "id"));
    }

    @Test
    public void testAppendNullDataValue() {
        LogData logData = newTestLogData("id", null);
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = null", outBuffer.toString());
    }

    @Test
    public void testAppendStringDataValue() {
        LogData logData = newTestLogData("id", "kltsjlfkgjskl");
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = 'kltsjlfkgjskl'", outBuffer.toString());

    }

    @Test
    public void testAppendShortDataValue() {
        LogData logData = newTestLogData("id", (short) 14);
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '14'", outBuffer.toString());

    }

    @Test
    public void testAppendIntegerDataValue() {
        LogData logData = newTestLogData("id", 145);
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '145'", outBuffer.toString());

    }

    @Test
    public void testAppendLongDataValue() {
        LogData logData = newTestLogData("id", 145L);
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '145'", outBuffer.toString());

    }

    @Test
    public void testAppendDoubleDataValue() {
        LogData logData = newTestLogData("id", 5.44564564564D);
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '5.44564564564'", outBuffer.toString());

    }

    @Test
    public void testAppendFloatDataValue() {
        LogData logData = newTestLogData("id", 5.412329F);
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '5.412329'", outBuffer.toString());

    }

    @Test
    public void testAppendByteDataValue() {
        LogData logData = newTestLogData("id", (byte) 10);
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '10'", outBuffer.toString());

    }

    @Test
    public void testAppendBigIntegerDataValue() {
        LogData logData = newTestLogData("id", new BigInteger("187645645646456164651453123445"));
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '187645645646456164651453123445'", outBuffer.toString());

    }

    @Test
    public void testAppendBigDecimalDataValue() {
        LogData logData = newTestLogData("id", new BigDecimal(
        "14442545641212315.454564564231232184178941321864131965461"));
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX
                + "id = '14442545641212315.454564564231232184178941321864131965461'",
                outBuffer.toString());

    }

    @Test
    public void testAppendCharDataValue() {
        LogData logData = newTestLogData("id", 'c');
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = 'c'", outBuffer.toString());

    }

    @Test
    public void testAppendBooleanDataValue() {
        LogData logData = newTestLogData("id", true);
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = 'true'", outBuffer.toString());

    }

    @Test
    public void testAppendEnumDataValue() {
        LogData logData = newTestLogData("id", TestEnum.ENUM_VAL1);
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = 'ENUM_VAL1'", outBuffer.toString());

    }

    @Test
    public void testAppendObjectDataValue() {
        LogData logData = newTestLogData("id", new Date(1));
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = 'Thu Jan 01 01:00:00 CET 1970'", outBuffer.toString());

    }
    
    
    @Test
    public void testAppendStringArrayDataValue() {
        LogData logData = newTestLogData("id",  new String[] { "s1", "s2", "s3" });
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = 's1, s2, s3'", outBuffer.toString());
    }


    @Test
    public void testAppendIntArrayDataValue() {
        LogData logData = newTestLogData("id",  new int[] { 0, 1, 2 });
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '0, 1, 2'", outBuffer.toString());
    }


    @Test
    public void testAppendIntegerArrayDataValue() {
        LogData logData = newTestLogData("id",  new Integer[] { 0, 1, 2 });
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '0, 1, 2'", outBuffer.toString());
    }
    
    @Test
    public void testAppendObjectArrayDataValue() {
        LogData logData = newTestLogData("id",  new Object[] { 0, 1, 2 });
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '0, 1, 2'", outBuffer.toString());
    }
    
    @Test
    public void testAppendObjectArrayWithNullDataValue() {
        LogData logData = newTestLogData("id",  new Object[] { 0, null, 2 });
        StringBuilder outBuffer = new StringBuilder(TEST_PREFIX);
        BasicLogDataAppender appender = new BasicLogDataAppender();
        appender.append(outBuffer, logData, "id");
        assertEquals(TEST_PREFIX + "id = '0, null, 2'", outBuffer.toString());
    }
    
}
