/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2018 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.logger;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.Locale;
import java.util.logging.LogRecord;

import org.junit.Before;
import org.junit.Test;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;

import com.ebmwebsourcing.easycommons.time.SimulatedTime;

public class LogDataFormatterTest extends AbstractLogDataFormatterTest {

    protected static final String INFORMATIONAL_MESS_PART = "This is a test message";

    protected SimulatedTime time = null;

    protected long utcTime = 0L;

    protected String formattedDate = null;

    @Before
    public void initSimulatedTime() {
        this.time = new SimulatedTime();
        // Stop the time (call me god )
        this.time.pause();
        this.utcTime = this.time.getUtcTime();
        this.formattedDate = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss,SSS 'GMT'Z",
                Locale.getDefault()).format(this.utcTime);
    }

    @Test
    public final void testNoLogDataAppenderPresentByDefault() {
        LogDataFormatter formatter = new LogDataFormatter();
        assertEquals(0, formatter.getLogDataAppenders().length);
    }

    @Test
    public final void testAddDataAppender() {
        LogDataFormatter formatter = new LogDataFormatter();
        BasicLogDataAppender blda1 = new BasicLogDataAppender();
        BasicLogDataAppender blda2 = new BasicLogDataAppender();
        formatter.addLogDataAppender(blda1);
        formatter.addLogDataAppender(blda2);
        assertArrayEquals(new LogDataAppender[] { blda1, blda2 }, formatter.getLogDataAppenders());
    }

    @Test
    public final void testFormatMessageWithoutLogData() throws Exception {
        LogDataFormatter formatter = new LogDataFormatter();
        String msg = "This is a test {0} message";
        LogRecord logRecord = LogRecordHelper.newLogRecord(Level.INFO, msg, this.utcTime);
        logRecord.setParameters(new Object[] { "REPLACED" });
        String myLog = formatter.formatMessage(logRecord);
        assertEquals("This is a test REPLACED message", myLog);
    }

    @Test
    public final void testFormatMessageWithLogData() throws Exception {
        LogDataFormatter formatter = new LogDataFormatter();
        String msg = "This is a test {0} message";
        LogRecord logRecord = LogRecordHelper.newLogRecord(Level.INFO, msg, this.utcTime);
        logRecord.setParameters(new Object[] { createTestLogData() });
        String myLog = formatter.formatMessage(logRecord);
        assertEquals("This is a test {0} message", myLog);
    }

    @Test
    public final void testFormattingWithoutExceptionNoRecordParam() throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        testFormatting(formatterUnderTest, null, null, null, "");
    }

    @Test
    public final void testFormattingWithExceptionNoRecordParam() throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        Throwable thrown = new RuntimeException("error...");
        testFormatting(formatterUnderTest, null, thrown, null, "");
    }

    @Test
    public final void testFormattingWithExceptionChainNoRecordParam() throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        Throwable cause2 = new RuntimeException("error cause2...");
        Throwable cause1 = new RuntimeException("error cause1...", cause2);
        Throwable thrown = new RuntimeException("error...", cause1);
        testFormatting(formatterUnderTest, null, thrown, null, "");
    }

    @Test
    public final void testFormattingWithoutExceptionUnhandledRecordParam() throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        testFormatting(formatterUnderTest, new Object[] { 1, "fakeparam", 2.3 }, null, null, "");
    }

    @Test
    public final void testFormattingWithoutExceptionLogDataAsRecordParam() throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        formatterUnderTest.addLogDataAppender(new BasicLogDataAppender());
        testFormatting(formatterUnderTest, new Object[] { createTestLogData() }, null, null, 
                " {{key1 = 'value1', key2 = 'value2', key3 = 'value3'}}");
    }

    @Test
    public final void testFormattingWithoutExceptionEmptyLogDataAsRecordParam() throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        formatterUnderTest.addLogDataAppender(new BasicLogDataAppender());
        testFormatting(formatterUnderTest, new Object[] { new TestLogData() }, null, null, "");
    }

    @Test
    public final void testFormattingWithoutExceptionLogDataAsRecordParamButNoAppender()
            throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        testFormatting(formatterUnderTest, new Object[] { createTestLogData() }, null, null, "");
    }

    @Test
    public final void testFormattingWithExceptionLogDataAsRecordParam() throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        formatterUnderTest.addLogDataAppender(new BasicLogDataAppender());
        Throwable thrown = new RuntimeException("error...");
        testFormatting(formatterUnderTest, new Object[] { createTestLogData() }, thrown, null, 
                " {{key1 = 'value1', key2 = 'value2', key3 = 'value3'}}");
    }

    @Test
    public final void testFormatMessageWithNullAsParametersValue() throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        testFormatting(formatterUnderTest, null, null, null, "");
    }

    @Test
    public final void testFormatMessageWithNullAsFirstValueInParameters() throws Exception {
        LogDataFormatter formatterUnderTest = new LogDataFormatter();
        testFormatting(formatterUnderTest, new String[] { null }, null, null, "");
    }

    @Test
    public void testFormatMessageWithPrefix() {
    	LogDataFormatter formatterUnderTest = new LogDataFormatter();
    	String expectedPrefix ="petals-node1";
    	formatterUnderTest.setPrefix(expectedPrefix);
        testFormatting(formatterUnderTest, new Object[] { createTestLogData() }, null, expectedPrefix,
                "");
    }
    
    protected void testFormatting(LogDataFormatter formatterUnderTest, Object[] logDataParameters,
            Throwable thrown, String expectedPrefix, String expectedExtraParametersMessPart) {
        for (java.util.logging.Level level : Level.getLevels()) {
            LogRecord logRecord = thrown != null ? LogRecordHelper.newLogRecordWithException(level,
                    INFORMATIONAL_MESS_PART, this.utcTime, thrown) : LogRecordHelper.newLogRecord(
                    level, INFORMATIONAL_MESS_PART, this.utcTime);
            if (logDataParameters != null) {
                logRecord.setParameters(logDataParameters);
            }
            String expectedLogMessage = (expectedPrefix != null ? expectedPrefix + " " : "") 
            		+ this.formattedDate + " " + level + " ["
                    + logRecord.getLoggerName() + "] : " + INFORMATIONAL_MESS_PART
                    + expectedExtraParametersMessPart + "\n"
                    + (thrown != null ? generateExpectedExceptionMessagePart(thrown) : "");

            String myLog = formatterUnderTest.format(logRecord);

            assertEquals(expectedLogMessage, myLog);
        }
    }
    
    protected String generateExpectedExceptionMessagePart(Throwable throwable) {
        final StringWriter sw = new StringWriter();
        throwable.printStackTrace(new PrintWriter(sw));
        sw.flush();
        return sw.toString();
    }
}
