/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2018 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.logger;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;

import java.util.List;
import java.util.logging.ConsoleHandler;
import java.util.logging.FileHandler;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.junit.Test;

/**
 * @author aruffie
 * 
 */
public class LoggerHelperTest {

    @Test
    public void testGetLevelOrParentLevelOnChildOfRootLogger() {
        final Logger logger = Logger.getLogger("myChildLogger0");
        assertNotNull(LoggerHelper.getLevelOrParentLevel(logger));
    }

    @Test
    public void testGetLevelOrParentLevelOnChildOfUserLogger() {
        final Logger parentLogger = Logger.getLogger("myParentLogger1");
        parentLogger.setLevel(Level.FINEST);
        final Logger childLogger = Logger.getLogger("myParentLogger1.myChildLogger1");
        assertSame(parentLogger.getLevel(), LoggerHelper.getLevelOrParentLevel(childLogger));
    }

    @Test
    public void testGetLevelOrParentLevelOnLittleChildUserLogger() {
        final Logger parentLogger = Logger.getLogger("myParentLogger2");
        parentLogger.setLevel(Level.FINEST);
        final Logger littleChildLogger = Logger
                .getLogger("myParentLogger2.myChildLogger2.myLittleChildLogger2");
        assertSame(parentLogger.getLevel(), LoggerHelper.getLevelOrParentLevel(littleChildLogger));
    }

    @Test
    public void testGetLevelOrParentLevelOnChildOfUserLoggerWhenSpecificLevelGiven() {
        final Logger parentLogger = Logger.getLogger("myParentLogger3");
        final Logger childLogger = Logger.getLogger("myParentLogger3.myChild3");
        childLogger.setLevel(Level.SEVERE);
        parentLogger.setLevel(Level.INFO);
        assertEquals(Level.SEVERE, LoggerHelper.getLevelOrParentLevel(childLogger));
    }
    
    @Test
    public void testGetHandlersOrParentHanderslOnChildOfRootLogger() {
        final Logger rootLogger = Logger.getLogger("");
        final ConsoleHandler parentHandler = new ConsoleHandler();
        rootLogger.addHandler(parentHandler);
        final Logger logger = Logger.getLogger("myChildLogger0");
        assertTrue(LoggerHelper.getHandlersOrParentHanders(logger).contains(parentHandler));
    }
    
    @Test
    public void testGetHandlersOrParentHanderslOnALogger() {
        final Logger logger = Logger.getLogger("myChildLogger0");
        final ConsoleHandler parentHandler = new ConsoleHandler();
        logger.addHandler(parentHandler);
        assertTrue(LoggerHelper.getHandlersOrParentHanders(logger).contains(parentHandler));
    }
    
    @Test
    public void testGetHandlersOrParentHandersOnLittleChildUserLoggerWithUseParentHandlersToTrue() {
        final Logger parentLogger = Logger.getLogger("myParentLogger4");
        final ConsoleHandler parentHandler = new ConsoleHandler();
        parentLogger.addHandler(parentHandler);
        final Logger littleChildLogger = Logger
                .getLogger("myParentLogger4.myChildLogger2.myLittleChildLogger2");
        List<Handler> handlersOrParentHanders = LoggerHelper.getHandlersOrParentHanders(littleChildLogger);
        assertTrue(handlersOrParentHanders.contains(parentHandler));
    }
    
    @Test
    public void testGetHandlersOrParentHandersOnLittleChildUserLoggerWithUseParentHandlersToFalseAndNoHandlerOnChildLogger() {
        final Logger parentLogger = Logger.getLogger("myParentLogger5");
        final ConsoleHandler parentHandler = new ConsoleHandler();
        parentLogger.addHandler(parentHandler);
        final Logger childLogger = Logger
                .getLogger("myParentLogger5.myChildLogger2");
        childLogger.setUseParentHandlers(false);
        final Logger littleChildLogger = Logger
                .getLogger("myParentLogger5.myChildLogger2.myLittleChildLogger2");
        assertTrue(!LoggerHelper.getHandlersOrParentHanders(littleChildLogger).contains(parentHandler));
    }
    
    @Test
    public void testGetHandlersOrParentHandersOnLittleChildUserLoggerWithUseParentHandlersToFalseAndAHandlerOnChildLogger() throws Exception {
        final Logger parentLogger = Logger.getLogger("myParentLogger6");
        final ConsoleHandler parentHandler = new ConsoleHandler();
        parentLogger.addHandler(parentHandler);
        final Logger childLogger = Logger
                .getLogger("myParentLogger6.myChildLogger2");
        childLogger.setUseParentHandlers(false);
        final FileHandler childHandler = new FileHandler();
        childLogger.addHandler(childHandler);
        final Logger littleChildLogger = Logger
                .getLogger("myParentLogger6.myChildLogger2.myLittleChildLogger2");
        assertTrue(LoggerHelper.getHandlersOrParentHanders(littleChildLogger).contains(childHandler));
    }
}
