/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2018 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.logger;

import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

/**
 * Allow to test and check the correct functionalities of {@link TestHandler} class
 * 
 * @author aruffie
 * 
 */
public class TestHandlerTest {

    // Published message prefix constant
    private final String LOG_RECORD_MSG = "TestPublishLogRecord record: ";

    @Test
    public void testGetAllRecordsAfterInit() {
        final TestHandler testHandler = new TestHandler();

        assertTrue("TestHandler is empty after initialization",
                testHandler.getAllRecords().size() == 0);
    }

    @Test
    public void testGetAllRecords() {
        final TestHandler testHandler = new TestHandler();
        assertTrue("TestHandler isn't empty after initialization", testHandler.getAllRecords()
                .size() == 0);

        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "1"));

        assertTrue("TestHandler hasn't 1 log record", testHandler.getAllRecords().size() == 1);

        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "2"));

        assertTrue("TestHandler hasn't 2 log records", testHandler.getAllRecords().size() == 2);

        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "3"));

        assertTrue("TestHandler hasn't 3 log records", testHandler.getAllRecords().size() == 3);

        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "4"));

        assertTrue("TestHandler hasn't 4 log records", testHandler.getAllRecords().size() == 4);

        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "5"));

        assertTrue("TestHandler hasn't 5 log records", testHandler.getAllRecords().size() == 5);
    }

    @Test
    public void testGetAllRecordsForLevel() {
        final TestHandler testHandler = new TestHandler();

        assertTrue("TestHandler is empty after initialization",
                testHandler.getAllRecords().size() == 0);

        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "1"));
        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "2"));
        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "3"));
        testHandler.publish(new LogRecord(Level.FINEST, LOG_RECORD_MSG + "4"));
        testHandler.publish(new LogRecord(Level.FINEST, LOG_RECORD_MSG + "5"));
        testHandler.publish(new LogRecord(Level.SEVERE, LOG_RECORD_MSG + "6"));

        assertTrue("3 records aren't recovered for INFO log level",
                testHandler.getAllRecords(Level.INFO).size() == 3);
        assertTrue("2 records aren't recovered for INFO log level",
                testHandler.getAllRecords(Level.FINEST).size() == 2);
        assertTrue("1 records aren't recovered for INFO log level",
                testHandler.getAllRecords(Level.SEVERE).size() == 1);
    }

    @Test
    public void testClearRecords() {
        final TestHandler testHandler = new TestHandler();

        assertTrue("TestHandler isn't empty after initialization", testHandler.getAllRecords()
                .size() == 0);

        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "1"));

        assertTrue("TestHandler hasn't one log record", testHandler.getAllRecords().size() == 1);

        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "2"));

        assertTrue("TestHandler hasn't two log record", testHandler.getAllRecords().size() == 2);

        testHandler.clearRecords();

        assertTrue("TestHandler isn't empty after cleaning",
                testHandler.getAllRecords().size() == 0);
    }

    @Test
    public void testPublishLogRecord() {
        final TestHandler testHandler = new TestHandler();

        assertTrue("TestHandler isn't empty after initialization", testHandler.getAllRecords()
                .size() == 0);

        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "1"));

        assertTrue("TestHandler hasn't one log record", testHandler.getAllRecords().size() == 1);
        assertTrue(LOG_RECORD_MSG + " uncorrectly published", testHandler.getAllRecords().get(0)
                .getMessage().equals(LOG_RECORD_MSG + "1"));
    }

    @Test
    public void testPublishLogRecordLevelOFF() {
        final TestHandler testHandler = new TestHandler();
        testHandler.setLevel(Level.OFF);
        testHandler.publish(new LogRecord(Level.INFO, LOG_RECORD_MSG + "1"));
        assertTrue(testHandler.getAllRecords().isEmpty());
    }
    
    @Test
    public void testGetLoggerWithTestHandler() {
        final TestHandler testHandler = new TestHandler();
        Logger logger = TestHandler.getLoggerWithTestHandler("testLogger", testHandler);
        
        assertEquals("testLogger", logger.getName());
        assertEquals(1, logger.getHandlers().length);
        assertEquals(testHandler, logger.getHandlers()[0]);
    }

}
