/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2018 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.logger.handler;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CyclicBarrier;
import java.util.logging.ErrorManager;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;
import java.util.logging.SimpleFormatter;

import org.junit.Before;
import org.junit.Test;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertSame;
import static org.junit.Assert.assertTrue;

import com.ebmwebsourcing.easycommons.io.FileSystemHelper;
import com.ebmwebsourcing.easycommons.lang.UncheckedException;
import com.ebmwebsourcing.easycommons.logger.LogDataFormatter;
import com.ebmwebsourcing.easycommons.logger.LogRecordHelper;
import com.ebmwebsourcing.easycommons.thread.ExecutionContext;
import com.ebmwebsourcing.easycommons.thread.TestThread;

public class ContextualFileHandlerTest {

    @Before
    public void before() {
        LogManager.getLogManager().reset();
        ExecutionContext.getProperties().clear();
    }

    @Test
    public void testConfigurationWithAllPropertiesSet() throws Exception {
        LogManager manager = LogManager.getLogManager();
        InputStream is = Thread.currentThread().getContextClassLoader()
                .getResourceAsStream("logConf.properties");
        manager.readConfiguration(is);

        ContextualFileHandler cfh = new ContextualFileHandler();
        assertSame(Level.FINEST, cfh.getLevel());
        assertEquals("my/test/basedir", cfh.getBasedir());
        assertEquals("my/test/subdir", cfh.getSubdir());
        assertEquals("myTestLogFilename", cfh.getLogFilename());
        assertEquals(LogDataFormatter.class, cfh.getFormatter().getClass());
    }

    @Test
    public void testDefaultConfiguration() throws Exception {
        ContextualFileHandler cfh = new ContextualFileHandler();
        assertSame(Level.INFO, cfh.getLevel());
        assertEquals(System.getProperty("user.dir"), cfh.getBasedir());
        assertEquals("", cfh.getSubdir());
        assertEquals(ContextualFileHandler.DEFAULT_LOG_FILE_NAME, cfh.getLogFilename());
        assertEquals(SimpleFormatter.class, cfh.getFormatter().getClass());
    }

    @Test(expected = UncheckedException.class)
    public void testConstructionWithEmptyLogFilename() throws Exception {
        new ContextualFileHandler("baseDirTest", "subDirTest", "", new Properties());
    }

    @Test(expected = UncheckedException.class)
    public void testConstructionWithFileSeparatorInLogFilename() throws Exception {
        new ContextualFileHandler("baseDirTest", "subDirTest", "test/Filename", new Properties());
    }

    @Test(expected = UncheckedException.class)
    public void testConstructionWithLoopPropertyInBasedir() throws Exception {
        Properties properties = new Properties();
        properties.put("test.loop1", "testBaseDir/${test.loop1}");
        new ContextualFileHandler("baseDirTest/${test.loop1}", "subDirTest", "logFileTest",
                properties);
    }

    @Test
    public void testConstructionWithResolvablePropertyInBasedir() throws Exception {
        Properties properties = new Properties();
        properties.put("test.resolvable", "testBaseDirValue");
        ContextualFileHandler cfh = new ContextualFileHandler("baseDirTest/${test.resolvable}",
                "subDirTest", "logFileTest", properties);
        assertEquals("baseDirTest/testBaseDirValue", cfh.getBasedir());
    }

    @Test
    public void testConstructionWithUnresolvedPropertyInBasedir() throws Exception {
        ContextualFileHandler cfh = new ContextualFileHandler("baseDirTest/${unresolved_property}",
                "subDirTest", "logFileTest", new Properties());
        assertEquals("baseDirTest/${unresolved_property}", cfh.getBasedir());
    }

    private static final void compareLogRecordsWithFormattedLogRecords(
            List<LogRecord> expectedLogRecords, List<String> formattedLogRecords) {
        String[] expectedFormattedLogRecords = new String[expectedLogRecords.size()];
        SimpleFormatter sf = new SimpleFormatter();
        for (int i = 0; i < expectedLogRecords.size(); ++i) {
            expectedFormattedLogRecords[i] = sf.format(expectedLogRecords.get(i));
        }
        assertArrayEquals(expectedFormattedLogRecords,
                formattedLogRecords.toArray(new String[formattedLogRecords.size()]));
    }

    private static final List<LogRecord> createTestLogRecords(String[] logMessages) {
        List<LogRecord> expectedLogRecords = new ArrayList<LogRecord>();
        for (String logMessage : logMessages) {
            LogRecord lr = LogRecordHelper.newLogRecord(Level.SEVERE, logMessage, 0);
            expectedLogRecords.add(lr);
        }
        return Collections.unmodifiableList(expectedLogRecords);
    }

    private static final void testPublishRecords(ContextualFileHandler cfh,
            List<LogRecord> logRecordsToPublish, File expectedOutputFile,
            List<LogRecord> expectedLogRecords) throws IOException {

        for (LogRecord lr : logRecordsToPublish) {
            cfh.publish(lr);
        }

        compareLogRecordsWithFormattedLogRecords(expectedLogRecords,
                parseFormattedRecords(expectedOutputFile));
    }

    @Test
    public void testPublishWithUnresolvedPropertyInSubdir() throws Exception {
        File tempdir = FileSystemHelper.createTempDir();
        File basedir = new File(tempdir, "baseDirTest/dirTest");

        ContextualFileHandler cfh = new ContextualFileHandler(basedir.getAbsolutePath(),
                "subDirTest/${unresolved_property}", "logFileTest", new Properties());

        testPublishRecords(cfh, createTestLogRecords(new String[] { "testMessage1" }), new File(
                basedir, "subDirTest/${unresolved_property}/logFileTest"),
                createTestLogRecords(new String[] { "testMessage1" }));
    }

    @Test
    public void testPublishWithUnresolvedPropertyInLogFilename() throws Exception {
        File tempdir = FileSystemHelper.createTempDir();
        File basedir = new File(tempdir, "baseDirTest/dirTest");

        ContextualFileHandler cfh = new ContextualFileHandler(basedir.getAbsolutePath(),
                "subDirTest/testDir", "${unresolved_property}", new Properties());

        testPublishRecords(cfh, createTestLogRecords(new String[] { "testMessage1" }), new File(
                basedir, "subDirTest/testDir/${unresolved_property}"),
                createTestLogRecords(new String[] { "testMessage1" }));
    }

    @Test
    public void testPublishButInvalidLogFilePath() throws Exception {
        File tempdir = FileSystemHelper.createTempDir();
        File basedir = new File(tempdir, "baseDirTest/dirTest");
        String subdir = "subDirTest/dirTest";
        String logFilename = "testlogFileName";

        testPublishWithReportedError(basedir, subdir, logFilename);
    }

    private void testPublishWithReportedError(File basedir, String subdir, String logFilename) {
        File alreadyExistingDir = new File(basedir, subdir + File.separator + logFilename);
        alreadyExistingDir.mkdirs();

        ContextualFileHandler cfh = new ContextualFileHandler(basedir.getAbsolutePath(), subdir,
                logFilename, new Properties());

        LogRecord lr = new LogRecord(Level.SEVERE, "testMessage1");

        final List<String> reportedErrors = new ArrayList<String>();
        cfh.setErrorManager(new ErrorManager() {
            @Override
            public synchronized void error(String msg, Exception ex, int code) {
                reportedErrors.add(msg);
            }
        });

        cfh.publish(lr);
        assertEquals(1, reportedErrors.size());
    }

    @Test
    public void testPublishWithLoopPropertyInSubdir() throws Exception {
        File tempdir = FileSystemHelper.createTempDir();
        File basedir = new File(tempdir, "baseDirTest/dirTest");
        String subdir = "subDirTest/${test.loop1}";
        String logFilename = "testlogFileName";

        ExecutionContext.getProperties().setProperty("test.loop1", "testBaseDir/${test.loop1}");

        testPublishWithReportedError(basedir, subdir, logFilename);
    }

    @Test
    public void testPublishWithLoopPropertyInFilename() throws Exception {
        File tempdir = FileSystemHelper.createTempDir();
        File basedir = new File(tempdir, "baseDirTest/dirTest");
        String subdir = "subDirTest/testSubDir";
        String logFilename = "test${test.loop1}";

        ExecutionContext.getProperties().setProperty("test.loop1", "testBaseDir/${test.loop1}");

        testPublishWithReportedError(basedir, subdir, logFilename);
    }

    @Test
    public void testPublishCreatesLogFileIfAbsent() throws Exception {
        File tempdir = FileSystemHelper.createTempDir();
        File basedir = new File(tempdir, "baseDirTest/dirTest");
        String subdir = "subDirTest/testSubDir";
        String logFilename = "testlogFileName";

        ContextualFileHandler cfh = new ContextualFileHandler(basedir.getAbsolutePath(), subdir,
                logFilename, new Properties());

        File logFile = new File(basedir, subdir + File.separator + logFilename);

        testPublishRecords(cfh, createTestLogRecords(new String[] { "testMessage1" }), new File(
                basedir, subdir + File.separator + logFilename),
                createTestLogRecords(new String[] { "testMessage1" }));

        assertTrue(logFile.delete());

        testPublishRecords(cfh,
                createTestLogRecords(new String[] { "testMessage2", "testMessage3" }), new File(
                        basedir, subdir + File.separator + logFilename),
                createTestLogRecords(new String[] { "testMessage2", "testMessage3" }));

    }

    @Test
    public void testPublishAppendsInLogFileIfPresentBetweenTwoRuns() throws Exception {
        File tempdir = FileSystemHelper.createTempDir();
        File basedir = new File(tempdir, "baseDirTest/dirTest");
        String subdir = "subDirTest/testSubDir";
        String logFilename = "testlogFileName";

        ContextualFileHandler cfh1 = new ContextualFileHandler(basedir.getAbsolutePath(), subdir,
                logFilename, new Properties());

        testPublishRecords(cfh1, createTestLogRecords(new String[] { "testMessage1" }), new File(
                basedir, subdir + File.separator + logFilename),
                createTestLogRecords(new String[] { "testMessage1" }));

        ContextualFileHandler cfh2 = new ContextualFileHandler(basedir.getAbsolutePath(), subdir,
                logFilename, new Properties());

        testPublishRecords(
                cfh2,
                createTestLogRecords(new String[] { "testMessage2", "testMessage3" }),
                new File(basedir, subdir + File.separator + logFilename),
                createTestLogRecords(new String[] { "testMessage1", "testMessage2", "testMessage3" }));
    }

    @Test
    public void testPublishAppendsInLogFileIfPresent() throws Exception {
        File tempdir = FileSystemHelper.createTempDir();
        File basedir = new File(tempdir, "baseDirTest/dirTest");
        String subdir = "subDirTest/testSubDir";
        String logFilename = "testlogFileName";

        ContextualFileHandler cfh = new ContextualFileHandler(basedir.getAbsolutePath(), subdir,
                logFilename, new Properties());

        testPublishRecords(cfh, createTestLogRecords(new String[] { "testMessage1" }), new File(
                basedir, subdir + File.separator + logFilename),
                createTestLogRecords(new String[] { "testMessage1" }));

        testPublishRecords(
                cfh,
                createTestLogRecords(new String[] { "testMessage2", "testMessage3" }),
                new File(basedir, subdir + File.separator + logFilename),
                createTestLogRecords(new String[] { "testMessage1", "testMessage2", "testMessage3" }));
    }

    @Test
    public void testPublishOnSeveralDifferentFilesConcurrently() throws Exception {
        final File tempdir = FileSystemHelper.createTempDir();
        final File basedir = new File(tempdir, "baseDirTest/dirTest");

        final ContextualFileHandler cfh = new ContextualFileHandler(basedir.getAbsolutePath(),
                "${subDirTest}/testDir", "testlogFileName", new Properties());

        final int nbThreads = 100;
        final CyclicBarrier barrier = new CyclicBarrier(nbThreads);

        TestThread[] testThreads = new TestThread[nbThreads];
        for (int i = 0; i < testThreads.length; ++i) {
            testThreads[i] = new TestThread(new Runnable() {

                @Override
                public void run() {
                    final String threadId = String.valueOf(Thread.currentThread().getId());
                    try {
                        barrier.await();
                    } catch (InterruptedException ie) {
                        throw new UncheckedException(ie);
                    } catch (BrokenBarrierException bbe) {
                        throw new UncheckedException(bbe);
                    }
                    ExecutionContext.getProperties().setProperty("subDirTest", threadId);
                    File expectedOutputFile = new File(basedir, threadId
                            + "/testDir/testlogFileName");
                    try {
                        testPublishRecords(cfh, createTestLogRecords(new String[] {
                                "testMessageA" + threadId, "testMessageB" + threadId,
                                "testMessageC" + threadId }), expectedOutputFile,
                                createTestLogRecords(new String[] { "testMessageA" + threadId,
                                        "testMessageB" + threadId, "testMessageC" + threadId }));
                    } catch (IOException ioe) {
                        throw new UncheckedException(ioe);
                    }
                }
            });
            testThreads[i].start();
        }

        for (int i = 0; i < testThreads.length; ++i) {
            testThreads[i].joinExplosively();
        }
    }

    private static final List<String> parseFormattedRecords(File file) throws IOException {
        assert file != null;

        // This implementation is dirty because it supposes a log record
        // formatted by a java.util.logging.SimpleFormatter takes 2 lines

        List<String> records = new ArrayList<String>();

        String strLine = null;
        BufferedReader br = null;
        try {
            br = new BufferedReader(new InputStreamReader(new FileInputStream(file)));

            strLine = br.readLine();
            while (strLine != null) {
                String record = strLine + '\n';
                strLine = br.readLine();
                assert strLine != null : "A log record formatted by a java.util.logging.SimpleFormatter takes 2 lines.";
                record = record + strLine + '\n';
                records.add(record);
                strLine = br.readLine();
            }
        } finally {
            if (br != null) {
                br.close();
            }
        }

        return Collections.unmodifiableList(records);
    }

}
