/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.thread;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import org.junit.Before;
import org.junit.Test;

import com.ebmwebsourcing.easycommons.thread.InheritableThreadContext;

/**
 * @author Olivier Fabre - EBM WebSourcing
 */
public class InheritableThreadContextTest {

    @Before
    public void before() {
        InheritableThreadContext.getParameters().clear();
    }

    @Test
    public void testIsEmptyOnInit() throws Exception {
        assertTrue(InheritableThreadContext.getParameters().isEmpty());
    }

    @Test
    public void testIsEmptyAfterPut() throws Exception {
        InheritableThreadContext.getParameters().put("testKey1", "testValue1");
        assertFalse(InheritableThreadContext.getParameters().isEmpty());
    }

    @Test
    public void testGetWithUnknownKey() throws Exception {
        assertNull(InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
    }

    @Test
    public void testPutAndGet() throws Exception {
        InheritableThreadContext.getParameters().put("testKey1", "testValue1");
        assertEquals("testValue1", InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
    }

    @Test
    public void testPutTwiceAndGet() throws Exception {
        InheritableThreadContext.getParameters().put("testKey1", "testValue1");
        InheritableThreadContext.getParameters().put("testKey1", "testValue1");
        assertEquals("testValue1", InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
    }

    @Test
    public void testUpdateAndGet() throws Exception {
        InheritableThreadContext.getParameters().put("testKey1", "testValue1");
        assertEquals("testValue1", InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
        InheritableThreadContext.getParameters().put("testKey1", "testValue2");
        assertEquals("testValue2", InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
    }

    @Test
    public void testClear() throws Exception {
        InheritableThreadContext.getParameters().put("testKey1", "testValue1");
        InheritableThreadContext.getParameters().put("testKey1", "testValue2");
        InheritableThreadContext.getParameters().clear();
        assertTrue(InheritableThreadContext.getParameters().isEmpty());
    }

    @Test
    public void testReset() throws Exception {
        InheritableThreadContext.getParameters().put("testKey1", "testValue1");
        InheritableThreadContext.getParameters().put("testKey1", "testValue2");
        InheritableThreadContext.reset();
        assertTrue(InheritableThreadContext.getParameters().isEmpty());
    }

    @Test
    public void testPutAndGetAfterReset() throws Exception {
        InheritableThreadContext.getParameters().put("testKey1", "oldTestValue1");
        InheritableThreadContext.reset();
        InheritableThreadContext.getParameters().put("testKey1", "testValue1");
        assertEquals("testValue1", InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
    }
    
    @Test
    public void testPutInParentThreadClearInChildThreadAndGetInChildThread() throws Exception {
        TestThread tt1 = new TestThread(new Runnable() {
            @Override
            public void run() {
                InheritableThreadContext.getParameters().put("testKey1", "testParentValue1");
                TestThread tt2 = new TestThread(new Runnable() {
                    @Override
                    public void run() {
                        InheritableThreadContext.getParameters().put("testKey1", "testChildValue1");
                        assertEquals("testChildValue1", InheritableThreadContext.getParameters()
                                .getValueOrDefault("testKey1"));
                        InheritableThreadContext.getParameters().clear();
                        assertEquals("testParentValue1", InheritableThreadContext.getParameters()
                                .getValueOrDefault("testKey1"));
                    }
                });
                tt2.start();
                try {
                    tt2.joinExplosively();
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        });
        tt1.start();
        tt1.joinExplosively();
    }

    @Test
    public void testPutAndGetInSiblingThreads() throws Exception {
        TestThread tt1 = new TestThread(new Runnable() {
            @Override
            public void run() {
                InheritableThreadContext.getParameters().put("testKey1", "testValue1");
            }
        });
        tt1.start();
        tt1.joinExplosively();

        TestThread tt2 = new TestThread(new Runnable() {
            @Override
            public void run() {
                assertNull(InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
            }
        });
        tt2.start();
        tt2.joinExplosively();
    }

    @Test
    public void testPutInParentThreadAndGetInChildThread() throws Exception {
        TestThread tt1 = new TestThread(new Runnable() {
            @Override
            public void run() {
                InheritableThreadContext.getParameters().put("testKey1", "testValue1");
                TestThread tt2 = new TestThread(new Runnable() {
                    @Override
                    public void run() {
                        assertEquals("testValue1",
                                InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
                    }
                });
                tt2.start();
                try {
                    tt2.joinExplosively();
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        });
        tt1.start();
        tt1.joinExplosively();
    }

    @Test
    public void testPutInChildThreadAndGetInParentThread() throws Exception {
        TestThread tt1 = new TestThread(new Runnable() {
            @Override
            public void run() {
                TestThread tt2 = new TestThread(new Runnable() {
                    @Override
                    public void run() {
                        InheritableThreadContext.getParameters().put("testKey1", "testValue1");
                    }
                });
                tt2.start();
                try {
                    tt2.joinExplosively();
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
                assertNull(InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
            }
        });
        tt1.start();
        tt1.joinExplosively();
    }

    @Test
    public void testPutInParentThreadPutInChildThreadAndGetInBoth() throws Exception {
        TestThread tt1 = new TestThread(new Runnable() {
            @Override
            public void run() {
                InheritableThreadContext.getParameters().put("testKey1", "testParentValue1");
                TestThread tt2 = new TestThread(new Runnable() {
                    @Override
                    public void run() {
                        InheritableThreadContext.getParameters().put("testKey1", "testChildValue1");
                        assertEquals("testChildValue1", InheritableThreadContext.getParameters()
                                .getValueOrDefault("testKey1"));
                    }
                });
                tt2.start();
                try {
                    tt2.joinExplosively();
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
                assertEquals("testParentValue1",
                        InheritableThreadContext.getParameters().getValueOrDefault("testKey1"));
            }
        });
        tt1.start();
        tt1.joinExplosively();
    }

}
