/****************************************************************************
 * Copyright (c) 2010-2012, EBM WebSourcing - All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of the University of California, Berkeley nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE REGENTS AND CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ****************************************************************************/
 
package com.ebmwebsourcing.easycommons.thread;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Semaphore;

/**
 * A task executor that could runs simultaneously all registered tasks.
 * @author Olivier Fabre - EBM WebSourcing
 */
public class SimultaneousTaskExecutor {

    private List<WaitingTask> tasksToExecute = new ArrayList<WaitingTask>();

    /**
     * Register a new task to execute.
     * 
     * @param task
     *            a {@link Runnable} task.
     */
    public void registerTask(Runnable task) {
        this.tasksToExecute.add(new WaitingTask(task));
    }

    /**
     * Clear task list
     */
    public void clearRegisteredTasks() {
        this.tasksToExecute.clear();
    }

    /**
     * Execute all registered tasks simultaneously. Returns when all registered
     * tasks are done.
     * 
     * @throws InterruptedException
     */
    public void executeAllRegisteredTasks() throws InterruptedException {
        final Collection<TestThread> clientPool = new ArrayList<TestThread>();
        final Object starterMonitor = new Object();
        final Semaphore readyMonitor = new Semaphore(tasksToExecute.size());
        for (int i = 0; i < tasksToExecute.size(); i++) {
            readyMonitor.acquire();
            WaitingTask taskToExecute = tasksToExecute.get(i);
            taskToExecute.init(readyMonitor, starterMonitor);
            final TestThread thread = new TestThread(taskToExecute);
            clientPool.add(thread);
            thread.start();
        }

        // Wait that client threads begin to start
        readyMonitor.acquire(tasksToExecute.size());

        // Start all tasks simultaneously
        synchronized (starterMonitor) {
            starterMonitor.notifyAll();
        }

        // Wait the end of processing
        for (TestThread thread : clientPool) {
            thread.joinExplosively();
        }
    }
}
