/**
 * Copyright (c) 2010-2012 EBM WebSourcing, 2012-2016 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the New BSD License (3-clause license).
 *
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the New BSD License (3-clause license)
 * for more details.
 *
 * You should have received a copy of the New BSD License (3-clause license)
 * along with this program/library; If not, see http://directory.fsf.org/wiki/License:BSD_3Clause/
 * for the New BSD License (3-clause license).
 */
package com.ebmwebsourcing.easycommons.xml;

import static org.junit.Assert.assertTrue;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringWriter;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class XMLHelperTest {

    @Test
    public void testCreateStringFromNodeList() throws Exception {
        String message = "<ns0:mail xmlns:ns0=\"http://petals.ow2.org/components/mail/version-3\"><ns0:host>192.168.1.206</ns0:host><ns0:from>user2@devmail.com</ns0:from><ns0:reply>user2@devmail.com</ns0:reply><ns0:to>user2@devmail.com</ns0:to><ns0:subject>Integration mail test(GenericService). MESSAGE TYPE: XML</ns0:subject><ns0:body><customElt>[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload</customElt></ns0:body></ns0:mail>";
        String expected = "<customElt xmlns:ns0=\"http://petals.ow2.org/components/mail/version-3\">[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload</customElt>";
        
        Document doc = createDOMDocumentFromMessage(message);
        
        String result = XMLHelper.toString(findBodyNode(doc).getChildNodes());
        assertTrue(XMLComparator.isEquivalent(expected, result));
    }

    @Test
    public void testCreateStringFromNodeListWithIllegalXmlCharEscaped() throws Exception {
        String message = "<ns0:mail xmlns:ns0=\"http://petals.ow2.org/components/mail/version-3\"><ns0:host>192.168.1.206</ns0:host><ns0:from>user2@devmail.com</ns0:from><ns0:reply>user2@devmail.com</ns0:reply><ns0:to>user2@devmail.com</ns0:to><ns0:subject>Integration mail test(GenericService). MESSAGE TYPE: XML</ns0:subject><ns0:body>&lt;customElt&gt;[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload&lt;/customElt&gt;</ns0:body></ns0:mail>";
        String expected = "<customElt>[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload</customElt>";
        
        Document doc = createDOMDocumentFromMessage(message);
        
        String result = XMLHelper.toString(findBodyNode(doc).getChildNodes());
        assertTrue(XMLComparator.isEquivalent(expected, result));
    }

    @Test
    public void testCreateStringFromNodeListWithCDATA() throws Exception {
        String message = "<ns0:mail xmlns:ns0=\"http://petals.ow2.org/components/mail/version-3\">"
                + "<ns0:host>192.168.1.206</ns0:host><ns0:from>user2@devmail.com</ns0:from><ns0:reply>user2@devmail.com</ns0:reply><ns0:to>user2@devmail.com</ns0:to><ns0:subject>Integration mail test(GenericService). MESSAGE TYPE: XML</ns0:subject>"
                + "<ns0:body>"
                + "<![CDATA[<customElt>[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload</customElt>]]>"
                + "</ns0:body></ns0:mail>";
        String expected = "<customElt>[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload</customElt>";
        
        Document doc = createDOMDocumentFromMessage(message);
        
        String result = XMLHelper.toString(findBodyNode(doc).getChildNodes());
        assertTrue(XMLComparator.isEquivalent(expected, result));
    }
    
    @Test
    public void testCreateStringFromDOMDocument() throws Exception {
        String message = "<ns0:mail xmlns:ns0=\"http://petals.ow2.org/components/mail/version-3\">"
                + "<ns0:host>192.168.1.206</ns0:host><ns0:from>user2@devmail.com</ns0:from><ns0:reply>user2@devmail.com</ns0:reply><ns0:to>user2@devmail.com</ns0:to><ns0:subject>Integration mail test(GenericService). MESSAGE TYPE: XML</ns0:subject>"
                + "<ns0:body>"
                + "<![CDATA[<customElt>[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload</customElt>]]>"
                + "</ns0:body></ns0:mail>";
        
        Document doc = createDOMDocumentFromMessage(message);
        
        String result = XMLHelper.createStringFromDOMDocument(doc);
        assertTrue(XMLComparator.isEquivalent(message, result));
        // the end of the tag (?>) is omitted in the test because extra attributes can be added by the transformer
        assertTrue(result.startsWith("<?xml version=\"1.0\" encoding=\"UTF-8\""));
    }
    
    @Test
    public void testCreateStringFromDOMNode() throws Exception {
        String message = "<ns0:mail xmlns:ns0=\"http://petals.ow2.org/components/mail/version-3\"><ns0:host>192.168.1.206</ns0:host><ns0:from>user2@devmail.com</ns0:from><ns0:reply>user2@devmail.com</ns0:reply><ns0:to>user2@devmail.com</ns0:to><ns0:subject>Integration mail test(GenericService). MESSAGE TYPE: XML</ns0:subject><ns0:body>&lt;customElt&gt;[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload&lt;/customElt&gt;</ns0:body></ns0:mail>";
        String expected = "<ns0:body xmlns:ns0=\"http://petals.ow2.org/components/mail/version-3\">&lt;customElt&gt;[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload&lt;/customElt&gt;</ns0:body>";
        
        Document doc = createDOMDocumentFromMessage(message);
        
        String result = XMLHelper.createStringFromDOMNode(findBodyNode(doc));
        assertTrue(XMLComparator.isEquivalent(expected, result));
        assertTrue(!result.startsWith("<?xml version=\"1.0\" encoding=\"UTF-8\"?>"));
    }
    
    @Test
    public void testWriteDocument() throws Exception {
        String message = "<ns0:mail xmlns:ns0=\"http://petals.ow2.org/components/mail/version-3\"><ns0:host>192.168.1.206</ns0:host><ns0:from>user2@devmail.com</ns0:from><ns0:reply>user2@devmail.com</ns0:reply><ns0:to>user2@devmail.com</ns0:to><ns0:subject>Integration mail test(GenericService). MESSAGE TYPE: XML</ns0:subject><ns0:body>&lt;customElt&gt;[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload&lt;/customElt&gt;</ns0:body></ns0:mail>";
        
        Document doc = createDOMDocumentFromMessage(message);
        
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        XMLHelper.writeDocument(doc, baos);
        String result = new String(baos.toByteArray());
        assertTrue(XMLComparator.isEquivalent(message, result));
        // the end of the tag (?>) is omitted in the test because extra attributes can be added by the transformer
        assertTrue(result.startsWith("<?xml version=\"1.0\" encoding=\"UTF-8\""));
    }
    
    @Test
    public void testCreateDocumentFromString() throws Exception {
        String xml = "<ns0:mail xmlns:ns0=\"http://petals.ow2.org/components/mail/version-3\"><ns0:host>192.168.1.206</ns0:host><ns0:from>user2@devmail.com</ns0:from><ns0:reply>user2@devmail.com</ns0:reply><ns0:to>user2@devmail.com</ns0:to><ns0:subject>Integration mail test(GenericService). MESSAGE TYPE: XML</ns0:subject><ns0:body>&lt;customElt&gt;[#IMAP-PAYLOAD-XML]Sending mail with addressing in the message payload&lt;/customElt&gt;</ns0:body></ns0:mail>";
        
        Document document = XMLHelper.createDocumentFromString(xml);
        Source source = new DOMSource(document);
        StringWriter writer = new StringWriter();
        StreamResult result = new StreamResult(writer);
        
        Transformer transformer = null;
        try {
            transformer = Transformers.takeTransformer();
            transformer.transform(source, result);
        } finally {
            if (transformer != null) {
                Transformers.releaseTransformer(transformer);
            }
        }
        
        assertTrue(XMLComparator.isEquivalent(xml, writer.toString()));
    }
    
    @Test
    public void testFindChildRecursiveFound() throws Exception {
        String xml = "<test:directory xmlns:test=\"http://petals.ow2.org/test/\">" +
            "<test:person><test:name>dupont</test:name><test:firstName>roger</test:firstName><test:address><test:street>rue de la fontaine</test:street><test:town>Marseille</test:town></test:address></test:person>" +
            "<test:person><test:name>durand</test:name><test:firstName>bernard</test:firstName><test:address><test:street>route de Paris</test:street><test:town>Lyon</test:town></test:address></test:person>" +
            "<test:person><test:name>dupond</test:name><test:firstName>jean</test:firstName><test:address><test:street>rue de l'URSS</test:street><test:town>Lille</test:town></test:address></test:person>" +
            "</test:directory>";
        
        Document document = XMLHelper.createDocumentFromString(xml);
        Node node = XMLHelper.findChild(document, "http://petals.ow2.org/test/", "street", true);

        assertTrue(node != null);
        assertTrue(node.getNamespaceURI() != null);
        assertTrue(node.getNamespaceURI().equals("http://petals.ow2.org/test/"));
        assertTrue(node.getLocalName() != null);
        assertTrue(node.getLocalName().equals("street"));
        assertTrue(node.getTextContent() != null);
        assertTrue(node.getTextContent().equals("rue de la fontaine"));
    }
    
    @Test
    public void testFindChildRecursiveNotFound() throws Exception {
        String xml = "<test:directory xmlns:test=\"http://petals.ow2.org/test/\">" +
        	"<test:person><test:name>dupont</test:name><test:firstName>roger</test:firstName><test:address><test:street>rue de la fontaine</test:street><test:town>Marseille</test:town></test:address></test:person>" +
        	"<test:person><test:name>durand</test:name><test:firstName>bernard</test:firstName><test:address><test:street>route de Paris</test:street><test:town>Lyon</test:town></test:address></test:person>" +
        	"<test:person><test:name>dupond</test:name><test:firstName>jean</test:firstName><test:address><test:street>rue de l'URSS</test:street><test:town>Lille</test:town></test:address></test:person>" +
        	"</test:directory>";
        
        Document document = XMLHelper.createDocumentFromString(xml);
        Node node = XMLHelper.findChild(document, "http://petals.ow2.org/test/", "stret", true);

        assertTrue(node == null);
    }
    
    @Test
    public void testFindChildNotRecursiveNotFound() throws Exception {
        String xml = "<test:directory xmlns:test=\"http://petals.ow2.org/test/\">" +
            "<test:person><test:name>dupont</test:name><test:firstName>roger</test:firstName><test:address><test:street>rue de la fontaine</test:street><test:town>Marseille</test:town></test:address></test:person>" +
            "<test:person><test:name>durand</test:name><test:firstName>bernard</test:firstName><test:address><test:street>route de Paris</test:street><test:town>Lyon</test:town></test:address></test:person>" +
            "<test:person><test:name>dupond</test:name><test:firstName>jean</test:firstName><test:address><test:street>rue de l'URSS</test:street><test:town>Lille</test:town></test:address></test:person>" +
            "</test:directory>";
        
        Document document = XMLHelper.createDocumentFromString(xml);
        Node node = XMLHelper.findChild(document, "http://petals.ow2.org/test/", "street", false);

        assertTrue(node == null);
    }
    
    @Test
    public void testFindChildNotRecursiveFound() throws Exception {
        String xml = "<test:directory xmlns:test=\"http://petals.ow2.org/test/\">" +
            "<test:person><test:name>dupont</test:name><test:firstName>roger</test:firstName><test:address><test:street>rue de la fontaine</test:street><test:town>Marseille</test:town></test:address></test:person>" +
            "<test:person><test:name>durand</test:name><test:firstName>bernard</test:firstName><test:address><test:street>route de Paris</test:street><test:town>Lyon</test:town></test:address></test:person>" +
            "<test:person><test:name>dupond</test:name><test:firstName>jean</test:firstName><test:address><test:street>rue de l'URSS</test:street><test:town>Lille</test:town></test:address></test:person>" +
            "</test:directory>";
        
        Document document = XMLHelper.createDocumentFromString(xml);
        Node node = XMLHelper.findChild(document.getDocumentElement(), null, "person", false);

        assertTrue(node != null);
        assertTrue(node.getNamespaceURI() != null);
        assertTrue(node.getNamespaceURI().equals("http://petals.ow2.org/test/"));
        assertTrue(node.getLocalName() != null);
        assertTrue(node.getLocalName().equals("person"));
        assertTrue(node.getFirstChild() != null);
        assertTrue(node.getFirstChild().getTextContent() != null);
        assertTrue(node.getFirstChild().getTextContent().equals("dupont"));
    }
    
    private static final Document createDOMDocumentFromMessage(String message) throws SAXException, IOException {
        DocumentBuilder documentBuilder = null;
        
        Document doc = null;
        try {
            documentBuilder = DocumentBuilders.takeDocumentBuilder();
            doc = documentBuilder.parse(new ByteArrayInputStream(message.getBytes()));
        } finally {
            if (documentBuilder != null) {
                DocumentBuilders.releaseDocumentBuilder(documentBuilder);
            }
        }
        
        return doc;
    }
    
    private static final Node findBodyNode(Document doc) {
        Element rootElt = doc.getDocumentElement();
        NodeList rootChildNodes = rootElt.getChildNodes();
        for (int i = 0; i < rootChildNodes.getLength(); i++) {
            String n = rootChildNodes.item(i).getNodeName();
            if (n != null) {
                if (n.endsWith("body")) {
                    return rootChildNodes.item(i);
                }
            }
        }
        
        return null;
    }
}
