/**
 * Copyright (c) 2011-2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.listener.incoming;

import static org.ow2.petals.binding.soap.SoapConstants.Axis2.COMPONENT_CONTEXT_SERVICE_PARAM;
import static org.ow2.petals.binding.soap.SoapConstants.Axis2.CONSUMES_EXTENSIONS_SERVICE_PARAM;
import static org.ow2.petals.binding.soap.SoapConstants.Axis2.CONSUMES_SERVICE_PARAM;
import static org.ow2.petals.binding.soap.SoapConstants.Axis2.LOGGER_SERVICE_PARAM;
import static org.ow2.petals.binding.soap.SoapConstants.Axis2.PETALS_RECEIVER_SERVICE_PARAM;
import static org.ow2.petals.binding.soap.SoapConstants.Axis2.RAMPART_MODULE;
import static org.ow2.petals.binding.soap.SoapConstants.Axis2.WSDL_LOAD_SERVICE_PARAM;
import static org.ow2.petals.binding.soap.SoapConstants.SOAP.FAULT_SERVER;

import java.util.List;
import java.util.logging.Logger;

import javax.jbi.component.ComponentContext;
import javax.jbi.messaging.MessageExchange.Role;

import org.apache.axis2.AxisFault;
import org.apache.axis2.context.MessageContext;
import org.apache.axis2.deployment.util.Utils;
import org.apache.axis2.description.AxisEndpoint;
import org.apache.axis2.description.AxisModule;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.InOutAxisOperation;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.description.WSDL2Constants;
import org.apache.axis2.handlers.AbstractHandler;
import org.ow2.easywsdl.extensions.wsdl4complexwsdl.api.Description;
import org.ow2.easywsdl.wsdl.api.Binding;
import org.ow2.easywsdl.wsdl.api.BindingOperation;
import org.ow2.easywsdl.wsdl.api.Input;
import org.ow2.easywsdl.wsdl.api.Operation;
import org.ow2.easywsdl.wsdl.api.Part;
import org.ow2.easywsdl.wsdl.api.binding.BindingProtocol.SOAPMEPConstants;
import org.ow2.petals.binding.soap.util.SUPropertiesHelper;
import org.ow2.petals.binding.soap.util.WsdlHelper;
import org.ow2.petals.commons.log.PetalsExecutionContext;
import org.ow2.petals.component.framework.api.configuration.ConfigurationExtensions;
import org.ow2.petals.component.framework.jbidescriptor.generated.Consumes;

/**
 * Loads the operations from the WSDL on first request
 * 
 * @author vnoel
 *
 */
public class PetalsLoadOperationsFromWSDLHandler extends AbstractHandler {

    @Override
    public InvocationResponse invoke(MessageContext msgContext) throws AxisFault {

        AxisService service = msgContext.getAxisService();

        // if the service has been found with the previous handler
        if (service != null && service.getParameter(WSDL_LOAD_SERVICE_PARAM) != null) {
            // All service parameters read from the WSDL MUST be processed
            // as an ATOMIC manner, so in a critical section
            synchronized (service) {
                final Parameter wsdLoadParam = service.getParameter(WSDL_LOAD_SERVICE_PARAM);
                // if the WSDL has already been retrieved
                if (wsdLoadParam != null) {
                    Parameter consumesConfigParam = service.getParameter(CONSUMES_SERVICE_PARAM);
                    Consumes consumes = (Consumes) consumesConfigParam.getValue();
                    Parameter componentContextParam = service.getParameter(COMPONENT_CONTEXT_SERVICE_PARAM);
                    ComponentContext componentContext = (ComponentContext) componentContextParam.getValue();
                    Parameter loggerParam = service.getParameter(LOGGER_SERVICE_PARAM);
                    Logger logger = (Logger) loggerParam.getValue();
                    Parameter petalsReceiverParam = service.getParameter(PETALS_RECEIVER_SERVICE_PARAM);
                    PetalsReceiver petalsReceiver = (PetalsReceiver) petalsReceiverParam.getValue();

                    Description desc = WsdlHelper.getDescription(consumes, componentContext, logger);

                    if (desc != null) {
                        addAxisOperation(service, petalsReceiver, desc);

                        // update the endpoint data
                        updateEndpoint(service, msgContext);

                        // enable WS-Addressing if necessary
                        Parameter consumesExtensionsParam = service.getParameter(CONSUMES_EXTENSIONS_SERVICE_PARAM);
                        ConfigurationExtensions consumesExtensions = (ConfigurationExtensions) consumesExtensionsParam
                                .getValue();
                        if (SUPropertiesHelper.isWSAEnabled(consumesExtensions)) {
                            Parameter disableAddressingParam = service.getParameter(
                                    org.apache.axis2.addressing.AddressingConstants.DISABLE_ADDRESSING_FOR_IN_MESSAGES);
                            service.removeParameter(disableAddressingParam);
                        }

                        // enable WS-Security if necessary
                        if (SUPropertiesHelper.getModules(consumesExtensions).contains(RAMPART_MODULE)) {
                            AxisModule axisModule = service.getAxisConfiguration().getModule(RAMPART_MODULE);
                            service.engageModule(axisModule, service);
                        }

                        service.removeParameter(wsdLoadParam);
                        service.removeParameter(petalsReceiverParam);
                    } else {
                        String errorMessage = "WSDL description can not been retrieved from JBI endpoint";
                        org.ow2.petals.component.framework.logger.Utils.addMonitFailureTrace(logger,
                                PetalsExecutionContext.getFlowAttributes(), errorMessage, Role.CONSUMER);
                        throw new AxisFault(errorMessage, FAULT_SERVER);
                    }
                }
            }
        }

        return InvocationResponse.CONTINUE;
    }

    private void addAxisOperation(AxisService service, PetalsReceiver petalsReceiver, Description desc) {
        List<Binding> bindings = desc.getBindings();
        for (Binding binding : bindings) {
            List<BindingOperation> bindingOperations = binding.getBindingOperations();
            for (BindingOperation bindingOperation : bindingOperations) {
                final AxisOperation genericOperation = new InOutAxisOperation(bindingOperation.getQName());
                genericOperation.setSoapAction(bindingOperation.getSoapAction());

                String wsdl2Mep = getWSDL2Mep(bindingOperation);
                genericOperation.setMessageExchangePattern(wsdl2Mep);

                genericOperation.setMessageReceiver(petalsReceiver);
                service.addOperation(genericOperation);

                Operation operation = bindingOperation.getOperation();
                if (operation != null) {
                    Input input = operation.getInput();
                    if (input != null) {
                        List<Part> parts = input.getParts();
                        if (parts != null && parts.size() > 1) {
                            org.ow2.easywsdl.schema.api.Element firstElement = parts.get(0).getElement();
                            service.addMessageElementQNameToOperationMapping(firstElement.getQName(), genericOperation);
                        }
                    }
                }
            }
        }
    }

    private String getWSDL2Mep(BindingOperation operation) {
        SOAPMEPConstants soapMep = operation.getMEP();
        String wsdl2Mep;
        if (soapMep == SOAPMEPConstants.ONE_WAY) {
            wsdl2Mep = WSDL2Constants.MEP_URI_IN_ONLY;
        } else if (soapMep == SOAPMEPConstants.REQUEST_RESPONSE) {
            wsdl2Mep = WSDL2Constants.MEP_URI_IN_OUT;
        } else {
            wsdl2Mep = WSDL2Constants.MEP_URI_OUT_ONLY;
        }
        return wsdl2Mep;
    }

    private void updateEndpoint(AxisService service, MessageContext msgContext) throws AxisFault {
        service.getEndpoints().clear();
        Utils.addEndpointsToService(service);

        // necessary to avoid NPE
        AxisEndpoint axisEndpoint = (AxisEndpoint) msgContext.getProperty(WSDL2Constants.ENDPOINT_LOCAL_NAME);
        if (axisEndpoint != null) {
            axisEndpoint = service.getEndpoint(axisEndpoint.getName());
            msgContext.setProperty(WSDL2Constants.ENDPOINT_LOCAL_NAME, axisEndpoint);
        }
    }

}
