/**
 * Copyright (c) 2005-2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.listener.incoming.jetty;

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.http.HttpServlet;

import org.apache.axis2.AxisFault;
import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.engine.ListenerManager;
import org.mortbay.log.Log;
import org.ow2.petals.binding.soap.listener.incoming.SoapServerConfig;
import org.ow2.petals.binding.soap.listener.incoming.servlet.ListServicesServlet;
import org.ow2.petals.binding.soap.listener.incoming.servlet.SoapServlet;
import org.ow2.petals.binding.soap.listener.incoming.servlet.WelcomeServlet;

/**
 * Create a Jetty server that will be the Axis 2 transport listener. 
 * <p>
 * This Jetty server is used to replace the SimpleHTTPServer provided by Axis that have
 * quite bad performances.
 * </p>
 * 
 * @author Christophe Hamerling - EBM WebSourcing
 */
public class AxisServletServer {

    /**
     * The logger
     */
    private final Logger logger;

    /**
     * the Jetty server
     */
    private SoapServletServer server;

    /**
     * The component properties
     */
    private SoapServerConfig config;

    /**
     * The Jetty server statistics
     */
    private ServerStats stats;

    /**
     * Create a SOAP server
     * 
     * @param logger
     *            the logger
     * @param config
     *            the SOAP server configuration
     * @param configContext
     *            Axis 2 configuration context
     * @param probes
     *            Technical monitoring probes
     * @throws AxisFault
     *             impossible to set Jetty as SOAP transport listener
     */
    public AxisServletServer(final Logger logger, final SoapServerConfig config,
            final ConfigurationContext configContext, final IncomingProbes probes) throws AxisFault {

        this.config = config;
        this.logger = logger;
        this.stats = new ServerStats();

        final String restrictedIp;
        if (config.isRestricted()) {
            restrictedIp = config.getHostAddress();
        } else {
            restrictedIp = null;
        }

        final HTTPConfig httpConfig = new HTTPConfig(restrictedIp, config.getHttpPort(),
                config.getJettyAcceptors());

        final HTTPSConfig httpsConfig;
        if (config.isHttpsEnabled()) {
            HTTPSTruststoreConfig httpsTruststoreConfig = new HTTPSTruststoreConfig(
                    config.getHttpsTruststoreType(), config.getHttpsTruststoreFile(),
                    config.getHttpsTruststorePassword());
            HTTPSKeystoreConfig httpsKeystoreConfig = new HTTPSKeystoreConfig(
                    config.getHttpsKeystoreKeyPassword(), config.getHttpsKeystorePassword(),
                    config.getHttpsKeystoreFile(), config.getHttpsKeystoreType());

            httpsConfig = new HTTPSConfig(restrictedIp, config.getHttpsPort(),
                    config.getJettyAcceptors(), httpsTruststoreConfig, httpsKeystoreConfig,
                    config.isHttpsClientAuthEnabled());
        } else {
            httpsConfig = null;
        }

        ServletServerConfig ssc = new ServletServerConfig(config.getServicesMapping(),
                config.getServicesContext(), config.getJettyThreadMaxPoolSize(),
                config.getJettyThreadMinPoolSize(), httpConfig, httpsConfig);

        this.server = new SoapServletServer(ssc,
                new Servlets(
                        createSoapServicesDispatcherServlet(configContext, probes),
                        createSoapServicesListingServlet(configContext, probes),
                        createWelcomeServlet(probes)),
                probes,
                logger);
    }

    private HttpServlet createSoapServicesListingServlet(final ConfigurationContext configContext,
            final IncomingProbes probes) {
        return new ListServicesServlet(configContext, config, probes, this.logger);
    }

    private HttpServlet createWelcomeServlet(final IncomingProbes probes) {
        return new WelcomeServlet(config, stats, probes, this.logger);
    }

    private HttpServlet createSoapServicesDispatcherServlet(
            final ConfigurationContext configContext, final IncomingProbes probes) {
        return new SoapServlet(this.logger, configContext, this.stats, probes, this.config);
    }

    /**
     * Start the SOAP server
     * 
     * @throws AxisFault
     *             the SOAP server can not start
     */
    public void start() throws AxisFault {
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.info("Starting HTTP(S)/SOAP server...");
            this.logger.info(this.config.getHostToDisplay() + " - HTTP Port : "
                    + this.config.getHttpPort() + " - HTTPS Port : " + this.config.getHttpsPort()
                    + " - HTTP(S) server max poolsize : " + this.config.getJettyThreadMaxPoolSize()
                    + " - HTTP(S) server min poolsize : " + this.config.getJettyThreadMinPoolSize()
                    + " - HTTP(S) server acceptors size : " + this.config.getJettyAcceptors());
        }
        try {
            this.stats.setStartTime(System.currentTimeMillis());
            this.server.start();
        } catch (final Exception e) {
            if (logger.isLoggable(Level.SEVERE)) {
                this.logger.log(Level.SEVERE, "Can not start the HTTP(S)/SOAP server");
            }
            throw new AxisFault("Can not start the HTTP(S)/SOAP server", e);
        }
    }

    /**
     * Stop the SOAP server
     * 
     * @throws AxisFault
     *             the SOAP server can not stop
     */
    public void stop() throws AxisFault {
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.log(Level.INFO, "Stopping HTTP(S)/SOAP server...");
        }

        try {
            this.server.stop();
            this.stats.setStopTime(System.currentTimeMillis());
        } catch (final InterruptedException e) {
            throw AxisFault.makeFault(e);
        } catch (final Exception e) {
            throw AxisFault.makeFault(e);
        }
    }

    public void shutdown(final ConfigurationContext configContext) throws AxisFault {
        if (this.logger.isLoggable(Level.INFO)) {
            this.logger.log(Level.INFO, "Shuting down HTTP(S)/SOAP server...");
        }
        final ListenerManager listenerManager = configContext.getListenerManager();
        if (listenerManager != null) {
            listenerManager.destroy();
        }
        Log.setLog(null);
    }

    /**
     * Return if the SOAP server is running
     * 
     * @return true if the SOAP server is running, otherwise false
     */
    public boolean isRunning() {
        return this.server.isRunning();
    }

    /**
     * Redirect specified URI to the specified web service address
     * 
     * @param from
     *            The URI to redirect
     * @param to
     *            The serviceName or address of the destination service
     */
    public void addRedirect(String from, String to) {
        this.config.addRedirect(from,
                "/" + this.config.getServicesContext() + "/" + this.config.getServicesMapping()
                        + "/" + to);
    }

    /**
     * Remove the redirection
     * 
     * @param from
     *            the URL to redirect
     */
    public void removeRedirect(String from) {
        this.config.removeRedirect(from);
    }

    public void addServlet(HttpServlet servlet) {

    }

    public SoapServletServer getHttpServer() {
        return this.server;
    }
}
