/**
 * Copyright (c) 2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.listener.incoming.jetty;

import org.ow2.petals.probes.api.key.ProbeKey;

/**
 * <p>
 * A probe key identifying a service operation invocation:
 * <ol>
 * <li>path of the web-service (URL without scheme, server and port),</li>
 * <li>IP address of the web-service client,</li>
 * <li>web-service operation invoked</li>
 * </ol>
 * </p>
 * 
 * @author Christophe DENEUX (PetalsLink)
 * 
 */
public class IncomingServiceKey implements ProbeKey {

    /**
     * Path the web-service (URL without scheme, server and port) as
     * {@link String}
     */
	private final String webServicePath;

    /**
     * Web-service operation invoked.
     */
    private final String webServiceOperation;

	/**
	 * IP address of the web-service client as {@link String}
	 */
    private final String webServiceClient;

	/**
	 * For optimization reason, the string representation of the
	 * ServiceClientKey is computed at constructor level as it is immutable, and
	 * not each time it is needed.
	 */
	private final String[] keyAsStringArray;

    /**
     * Creates a new instance of ServiceClientKey
     * 
     * @param webService
     *            Path the web-service (URL without scheme, server and port)
     * @param The
     *            web-service operation invoked
     * @param webServiceClient
     *            IP address of the web-service client
     */
    public IncomingServiceKey(final String webServicePath, final String webServiceOperation,
	        final String webServiceClient) {
        this.webServicePath = webServicePath;
        this.webServiceOperation = webServiceOperation;
		this.webServiceClient = webServiceClient;
        this.keyAsStringArray = new String[] { this.webServicePath, this.webServiceOperation,
                this.webServiceClient };
    }

    /**
     * @return Path the web-service (URL without scheme, server and port) as
     *         {@link String}
     */
	public String getWebServicePath() {
        return this.webServicePath;
	}

    /**
     * @return The web-service operation as {@link String}
     */
    public String getWebServiceOperation() {
        return this.webServiceOperation;
    }

    /**
     * @return The IP address of the web-service client as {@link String}
     */
    public String getWebServiceClient() {
		return this.webServiceClient;
	}

	@Override
	public String toString() {
        return new StringBuilder("(").append(this.webServicePath).append(", ")
                .append(this.webServiceOperation).append(", ")
                .append(this.webServiceClient)
				.append(")").toString();
	}

	@Override
	public String[] toStringArray() {
		return this.keyAsStringArray;
	}

    @Override
    public String toReadableString() {
        return this.toString();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#hashCode()
     */
    // Generated by Eclipse
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((webServiceClient == null) ? 0 : webServiceClient.hashCode());
        result = prime * result
                + ((webServiceOperation == null) ? 0 : webServiceOperation.hashCode());
        result = prime * result + ((webServicePath == null) ? 0 : webServicePath.hashCode());
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Object#equals(java.lang.Object)
     */
    // Generated by Eclipse
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IncomingServiceKey)) {
            return false;
        }
        IncomingServiceKey other = (IncomingServiceKey) obj;
        if (webServiceClient == null) {
            if (other.webServiceClient != null) {
                return false;
            }
        } else if (!webServiceClient.equals(other.webServiceClient)) {
            return false;
        }
        if (webServiceOperation == null) {
            if (other.webServiceOperation != null) {
                return false;
            }
        } else if (!webServiceOperation.equals(other.webServiceOperation)) {
            return false;
        }
        if (webServicePath == null) {
            if (other.webServicePath != null) {
                return false;
            }
        } else if (!webServicePath.equals(other.webServicePath)) {
            return false;
        }
        return true;
    }
}