/**
 * Copyright (c) 2008-2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.listener.incoming.jetty;

import java.io.IOException;
import java.io.Writer;
import java.util.logging.Logger;

import javax.servlet.http.HttpServletRequest;

import org.mortbay.util.StringUtil;
import org.ow2.petals.probes.api.exceptions.ProbeNotStartedException;
import org.ow2.petals.probes.api.probes.CounterProbe;
import org.ow2.petals.probes.api.probes.GaugeProbe;

/**
 * A hanlder to replace the Jetty one in order to provide better messages.
 * @author Christophe HAMERLING - EBM WebSourcing
 */
public class PetalsErrorHandler extends org.mortbay.jetty.handler.ErrorHandler {

    /**
     * The probe counting the number of requests on an unknown servlet.
     */
    final protected CounterProbe probeUnknownServlet;

    /**
     * The probe measuring the allocated threads of the pool.
     */
    final protected GaugeProbe<Long, Long> probeHttpServerThreadPoolAllocatedThreads;

    /**
     * The probe measuring the idle threads of the pool.
     */
    final protected GaugeProbe<Long, Long> probeHttpServerThreadPoolIdleThreads;

    /**
     * The probe measuring the enqueued requests of the thread pool.
     */
    final protected GaugeProbe<Long, Long> probeHttpServerThreadPoolQueuedRequests;

    /**
     * The component logger
     */
    private final Logger logger;

    /**
     * @param showStacks
     *            True if stack traces are shown in the error pages
     * @param probes
     *            The technical monitoring probes
     */
    public PetalsErrorHandler(final boolean showStacks, final IncomingProbes probes, final Logger logger) {
        super();
        this.probeUnknownServlet = probes.probeUnknownServlet;
        this.probeHttpServerThreadPoolAllocatedThreads = probes.probeHttpServerThreadPoolAllocatedThreads;
        this.probeHttpServerThreadPoolIdleThreads = probes.probeHttpServerThreadPoolIdleThreads;
        this.probeHttpServerThreadPoolQueuedRequests = probes.probeHttpServerThreadPoolQueuedRequests;
        this.logger = logger;
        this.setShowStacks(showStacks);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.mortbay.jetty.handler.ErrorHandler#writeErrorPageMessage(javax.servlet.http.HttpServletRequest,
     *      java.io.Writer, int, java.lang.String, java.lang.String)
     */
    @Override
    protected void writeErrorPageMessage(final HttpServletRequest request, final Writer writer,
            final int code, final String message, final String uri) throws IOException {
        writer.write("<h1>PEtALS BC SOAP</h1>");
        writer.write("<p>");
        writer.write("There are no WebService available at this URI : " + uri);
        writer.write("</p>");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.mortbay.jetty.handler.ErrorHandler#writeErrorPageBody(javax.servlet.http.HttpServletRequest,
     *      java.io.Writer, int, java.lang.String, boolean)
     */
    @Override
    protected void writeErrorPageBody(final HttpServletRequest request, final Writer writer,
            final int code, final String message, final boolean showStacks) throws IOException {
        try {
            this.probeUnknownServlet.inc();
            this.probeHttpServerThreadPoolAllocatedThreads.pick();
            this.probeHttpServerThreadPoolIdleThreads.pick();
            this.probeHttpServerThreadPoolQueuedRequests.pick();
        } catch (final ProbeNotStartedException e) {
            this.logger
                    .warning("HTTP probes are not started. Values of probes could be incorrect.");
        }

        String uri = request.getRequestURI();
        if (uri != null) {
            uri = StringUtil.replace(uri, "&", "&amp;");
            uri = StringUtil.replace(uri, "<", "&lt;");
            uri = StringUtil.replace(uri, ">", "&gt;");
        }

        this.writeErrorPageMessage(request, writer, code, message, uri);
        if (showStacks) {
            this.writeErrorPageStacks(request, writer);
        }

        writer
                .write("<p><i><small><a href=\"http://petals.ow2.org/\">Powered by PEtALS SOAP Binding Component</a></small></i></p>");
        for (int i = 0; i < 20; i++) {
            writer.write("<br/>                                                \n");
        }
    }

}
