/**
 * Copyright (c) 2011-2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.listener.incoming.jetty;

import java.util.logging.Logger;

import javax.servlet.http.HttpServlet;

import org.mortbay.jetty.Server;
import org.mortbay.jetty.handler.HandlerCollection;
import org.mortbay.jetty.nio.SelectChannelConnector;
import org.mortbay.jetty.security.SslSelectChannelConnector;
import org.mortbay.jetty.servlet.Context;
import org.mortbay.jetty.servlet.ServletHolder;
import org.mortbay.log.Log;
import org.mortbay.thread.BoundedThreadPool;
import org.ow2.petals.binding.soap.SoapConstants;

import com.ebmwebsourcing.easycommons.lang.StringHelper;

/**
 * @author Adrien Ruffie - EBM WebSourcing
 */
public class SoapServletServer {

    protected static final String WELCOME_SERVLET_NAME = "WelcomeServlet";

    protected static final String SOAP_SERVICES_LISTING_SERVLET_NAME = "ServicesListServlet";

    protected static final String SOAP_SERVICES_DISPATCHER_SERVLET_NAME = "SoapServlet";

    final Server server;

    private Context soapContext;

    private Context welcomeContext;

    /**
     * The logger
     */
    protected Logger logger;

    private static final int HEADER_BUFFER_SIZE = 4096 * 4;

    /**
     * The Jetty thread pool. Size is configured from component configuration.
     */
    private BoundedThreadPool threadPool;

    static {
        System.setProperty("org.mortbay.log.class", JettyNullLogger.class.getName());
        Log.getLog();
    }

    /**
     * @param config
     *            The configuration of the servlet server in terms of ports,
     *            thread numbers, ...
     * @param servlets
     *            The servlets to deploy
     * @param probes
     *            The technical monitoring probes
     * @param logger
     *            A logger
     */
    public SoapServletServer(final ServletServerConfig config, final Servlets servlets,
            final IncomingProbes probes, final Logger logger) {

        assert servlets.soapServicesDispatcherServlet != null;
        assert servlets.soapServicesListingServlet != null;
        assert servlets.welcomeServlet != null;
        assert probes.probeUnknownServlet != null;
        assert probes.probeInformationServlet != null;
        assert probes.probeHttpServerThreadPoolAllocatedThreads != null;
        assert probes.probeHttpServerThreadPoolIdleThreads != null;
        assert probes.probeHttpServerThreadPoolQueuedRequests != null;
        assert probes.probeHttpRequestsInvocationsCount != null;
        assert probes.probeHttpRequestsInvocationsResponseTime != null;
        assert config != null;
        assert logger != null;

        this.logger = logger;
        Log.setLog(new JettyLogger(logger));

        this.server = new Server();

        HTTPConfig httpConfig = config.getHttpConfig();
        if (httpConfig != null) {
            this.server.addConnector(createNIOHTTPConnector(httpConfig.getHttpPort(),
                    httpConfig.getHttpRestrictedIP(), httpConfig.getAcceptorSize()));
        }

        HTTPSConfig httpsConfig = config.getHttpsConfig();
        if (httpsConfig != null) {
            SslSelectChannelConnector nioSslConnector = createNIOHTTPSConnector(
                    httpsConfig.getHttpsPort(), httpsConfig.getHttpsRestrictedIP(),
                    httpsConfig.getAcceptorSize());

            if(httpsConfig.getHttpsKeystoreConfig() != null) {
               this.initKeyStore(nioSslConnector, httpsConfig.getHttpsKeystoreConfig());
            }
            if(httpsConfig.getHttpsTruststoreConfig() != null) {
                this.initTrustStore(nioSslConnector, httpsConfig.getHttpsTruststoreConfig());
            }
            nioSslConnector.setNeedClientAuth(httpsConfig.isHttpsClientAuthEnabled());

            this.server.addConnector(nioSslConnector);
        }

        this.threadPool = new BoundedThreadPool();
        this.threadPool.setName("BCSoapJettyThreadPool");
        this.threadPool.setMaxThreads(config.getServerMaxPoolSize());
        this.threadPool.setMinThreads(config.getServerMinPoolSize());
        this.server.setThreadPool(this.threadPool);

        // create context handlers
        final HandlerCollection handlers = new HandlerCollection();

        // create contexts
        this.soapContext = this.createSoapContext(handlers, "/" + config.getServicesContext(),
                probes);
        this.welcomeContext = this.createWelcomeContext(handlers, "/");

        // deploy servlets
        this.deploySoapServicesDispatcherServlet(servlets.soapServicesDispatcherServlet,
                "/" + config.getServicesMapping() + "/*");
        this.deploySoapServicesListingServlet(servlets.soapServicesListingServlet,
                "/" + config.getServicesMapping() + "/" + SoapConstants.Component.MAPPING_NAME);
        this.deployWelcomeServlet(servlets.welcomeServlet, "/*");

        this.server.setHandler(handlers);
    }

    private Context createSoapContext(final HandlerCollection handlers, final String contextPath,
            final IncomingProbes probes) {
        final Context soapContext = new Context(handlers, contextPath, Context.SESSIONS);
        soapContext.setErrorHandler(new PetalsErrorHandler(false, probes, this.logger));
        return soapContext;
    }

    private Context createWelcomeContext(HandlerCollection handlers, String contextPath) {
        final Context welcomeContext = new Context(handlers, contextPath, Context.SESSIONS);
        return welcomeContext;
    }

    private void deploySoapServicesDispatcherServlet(
            final HttpServlet soapServicesDispatcherServlet,
            String path) {
        final ServletHolder soapServletHolder = new ServletHolder(soapServicesDispatcherServlet);
        soapServletHolder.setName(SOAP_SERVICES_DISPATCHER_SERVLET_NAME);
        this.soapContext.addServlet(soapServletHolder, path);
    }

    private void deploySoapServicesListingServlet(HttpServlet soapServicesListingServlet,
            String path) {
        final ServletHolder listServicesServletHolder = new ServletHolder(
                soapServicesListingServlet);
        listServicesServletHolder.setName(SOAP_SERVICES_LISTING_SERVLET_NAME);
        this.soapContext.addServlet(listServicesServletHolder, path);
    }

    private void deployWelcomeServlet(HttpServlet welcomeServlet, String path) {
        final ServletHolder welcomeServletHolder = new ServletHolder(welcomeServlet);
        welcomeServletHolder.setName(WELCOME_SERVLET_NAME);
        this.welcomeContext.addServlet(welcomeServletHolder, path);
    }

    private void initTrustStore(SslSelectChannelConnector nioSslConnector, HTTPSTruststoreConfig httpsTruststoreConfig) {
        // set truststore parameters
        String httpsTruststoreFile = httpsTruststoreConfig.getHttpsTruststoreFile();
        String httpsTruststorePassword = httpsTruststoreConfig.getHttpsTruststorePassword();
        String httpsTruststoreType = httpsTruststoreConfig.getHttpsTruststoreType();

        if (httpsTruststoreFile != null && httpsTruststorePassword != null) {
            nioSslConnector.setTruststore(httpsTruststoreFile);
            nioSslConnector.setTrustPassword(httpsTruststorePassword);
            nioSslConnector.setNeedClientAuth(true);
        }

        if (httpsTruststoreType != null) {
            nioSslConnector.setTruststoreType(httpsTruststoreType);
        }
    }

    private SslSelectChannelConnector createNIOHTTPSConnector(int httpsPort, String host,
            int acceptors) {
        SslSelectChannelConnector nioSslConnector = new SslSelectChannelConnector();
        if (!StringHelper.isNullOrEmpty(host)) {
            nioSslConnector.setHost(host);
        }
        nioSslConnector.setPort(httpsPort);
        nioSslConnector.setHeaderBufferSize(HEADER_BUFFER_SIZE);
        nioSslConnector.setStatsOn(false);
        nioSslConnector.setAcceptors(acceptors);

        return nioSslConnector;
    }

    private void initKeyStore(SslSelectChannelConnector nioSslConnector, HTTPSKeystoreConfig httpsKeystoreConfig) {
        // set keystore parameters
        nioSslConnector.setKeystore(httpsKeystoreConfig.getHttpsKeystoreFile());
        nioSslConnector.setPassword(httpsKeystoreConfig.getHttpsKeystorePassword());
        nioSslConnector.setKeyPassword(httpsKeystoreConfig.getHttpsKeystoreKeyPassword());
        String httpsKeystoreType = httpsKeystoreConfig.getHttpsKeystoreType();
        if (httpsKeystoreType != null) {
            nioSslConnector.setKeystoreType(httpsKeystoreType);
        }
    }

    private SelectChannelConnector createNIOHTTPConnector(int httpPort, String host,
            int jettyAcceptors) {
        // jetty http connector configuration
        SelectChannelConnector nioConnector = new SelectChannelConnector();
        nioConnector.setPort(httpPort);

        // If we assign the host, we will only be able to contact server
        // on it. No value or a null one is a wildcard so connection is possible
        // on network interface
        // @see java.net.InetSocketAddress
        if (!StringHelper.isNullOrEmpty(host)) {
            nioConnector.setHost(host);
        }

        nioConnector.setHeaderBufferSize(HEADER_BUFFER_SIZE);
        nioConnector.setStatsOn(false);
        nioConnector.setAcceptors(jettyAcceptors);
        return nioConnector;
    }

    public void start() throws Exception {
        this.server.start();
    }

    public void stop() throws Exception {
        this.server.stop();
    }

    public boolean isRunning() {
        return this.server.isRunning();
    }

    public BoundedThreadPool getThreadPool() {
        return this.threadPool;
    }
}
