/**
 * Copyright (c) 2007-2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.listener.incoming.servlet;

import static org.ow2.petals.binding.soap.SoapConstants.Axis2.OUTGOING_SERVICE_CLIENT_PREFIX;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServlet;

import org.apache.axis2.context.ConfigurationContext;
import org.apache.axis2.description.AxisService;
import org.ow2.petals.binding.soap.listener.incoming.SoapServerConfig;
import org.ow2.petals.binding.soap.listener.incoming.jetty.IncomingProbes;
import org.ow2.petals.probes.api.exceptions.ProbeNotStartedException;
import org.ow2.petals.probes.api.probes.CounterProbe;
import org.ow2.petals.probes.api.probes.GaugeProbe;

/**
 * Servlet used to display the list of services. 
 * <p>
 * It replaces the Axis2 one.
 * </p>
 * 
 * @author Christophe Hamerling - EBM WebSourcing
 */
public class ListServicesServlet extends HttpServlet {

	private static final long serialVersionUID = -4951673084170985493L;

	/**
	 * The HTML title string
	 */
	private static final String HTML_TOP = "<html><head><title>petals-bc-soap : Services List</title></head><body><h1>Petals BC SOAP</h1>";

	private static final String HTML_BOTTOM = "</body></html>";

	/**
	 * The axis2 configuration context used to get services list
	 */
	private transient final ConfigurationContext configContext;

	/**
	 * The SOAP server configuration
	 */
	private transient final SoapServerConfig soapServerConfig;

    /**
     * The probe counting unknown URLs received by the Jetty server.
     */
    private transient final CounterProbe probeInformationServlet;

    /**
     * The probe counting allocated threads in the thread pool of the HTTP
     * server.
     */
    private transient GaugeProbe<Long, Long> probeHttpServerThreadPoolAllocatedThreads;

    /**
     * The probe counting idle threads in the thread pool of the HTTP server.
     */
    private transient GaugeProbe<Long, Long> probeHttpServerThreadPoolIdleThreads;

    /**
     * The probe counting requests in the queue of the thread pool of the HTTP
     * server.
     */
    private transient GaugeProbe<Long, Long> probeHttpServerThreadPoolQueuedRequests;

    /**
     * The component logger
     */
    private transient final Logger logger;

	/**
     * Creates a new instance of ListServicesServlet
     * 
     * @param configurationContext
     * @param active
     * @param probes
     *            Technical monitoring probes
     * @param logger
     *            The component logger
     */
    public ListServicesServlet(final ConfigurationContext configurationContext,
            final SoapServerConfig soapConfig, final IncomingProbes probes, final Logger logger) {
		this.configContext = configurationContext;
		this.soapServerConfig = soapConfig;
        this.logger = logger;
        this.probeHttpServerThreadPoolAllocatedThreads = probes.probeHttpServerThreadPoolAllocatedThreads;
        this.probeHttpServerThreadPoolIdleThreads = probes.probeHttpServerThreadPoolIdleThreads;
        this.probeHttpServerThreadPoolQueuedRequests = probes.probeHttpServerThreadPoolQueuedRequests;
        this.probeInformationServlet = probes.probeInformationServlet;
	}

	/**
	 * The list has not been activated in component
	 * 
	 * @param out
	 * @throws IOException
	 */
	private void listNotAvailable(final ServletOutputStream out) throws IOException {
		out.write("<h1><font color='red'>The list of services is not available</font></h1>".getBytes());
		out.write("It must be activated in the SOAP component descriptor...".getBytes());
		out.write("</body></html>".getBytes());
	}

	/**
	 * Print the list of services with links to their WSDL description
	 * 
	 * @param out
	 * @throws IOException
	 */
	private void printServicesList(final ServletOutputStream out, String transport) throws IOException {
		final Map<?, ?> services = this.configContext.getAxisConfiguration().getServices();
		final Set<?> keys = services.keySet();

		out.write("<h2>Available services</h2>".getBytes());
		boolean atLeastOneService = false;
		if (keys.size() > 0) {
			for (final Object key : keys) {
				final AxisService service = (AxisService) services.get(key);
				if (!service.getName().startsWith(OUTGOING_SERVICE_CLIENT_PREFIX)) {
				    final List<String> exposedTransports = service.getExposedTransports();
				    if(exposedTransports.contains(transport)) {
				        atLeastOneService = true;
    					out.write("<li><a href='".getBytes());
    					out.write(service.getName().getBytes());
    					out.write("?wsdl'>".getBytes());
    					out.write(service.getName().getBytes());
    					out.write("</a></li>".getBytes());
				    }
				}
			}
		}
		
		if(!atLeastOneService) {
			out.write("No service".getBytes());
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.servlet.http.HttpServlet#service(javax.servlet.ServletRequest,
	 * javax.servlet.ServletResponse)
	 */
	@Override
	public void service(final ServletRequest request, final ServletResponse response) throws ServletException, IOException {
        try {
            this.probeInformationServlet.inc();
            this.probeHttpServerThreadPoolAllocatedThreads.pick();
            this.probeHttpServerThreadPoolIdleThreads.pick();
            this.probeHttpServerThreadPoolQueuedRequests.pick();
        } catch (final ProbeNotStartedException e) {
            this.logger
                    .warning("HTTP probes are not started. Values of probes could be incorrect.");
        }

        final ServletOutputStream out = response.getOutputStream();
		out.write(HTML_TOP.getBytes());

        if (this.soapServerConfig.isProvidesList()) {
            this.printServicesList(out, request.getScheme());
		} else {
            this.listNotAvailable(out);
		}

		out.write(("<p><center><a href='" + this.soapServerConfig.getBaseURL(request.getScheme()) + "'> - Index -</a></center></p>").getBytes());
		out.write(HTML_BOTTOM.getBytes());
		out.flush();
		out.close();
	}

}
