/**
 * Copyright (c) 2008-2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.listener.incoming.servlet;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.logging.Logger;

import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.mortbay.jetty.Server;
import org.ow2.petals.binding.soap.SoapConstants;
import org.ow2.petals.binding.soap.listener.incoming.SoapServerConfig;
import org.ow2.petals.binding.soap.listener.incoming.jetty.IncomingProbes;
import org.ow2.petals.binding.soap.listener.incoming.jetty.ServerStats;
import org.ow2.petals.probes.api.exceptions.ProbeNotStartedException;
import org.ow2.petals.probes.api.probes.CounterProbe;
import org.ow2.petals.probes.api.probes.GaugeProbe;

/**
 * A servlet which displays basic information. 
 * <p>
 * It replaces the default 404 pages.
 * </p>
 * 
 * @author Christophe HAMERLING - EBM WebSourcing
 */
public class WelcomeServlet extends HttpServlet {

    private static final long serialVersionUID = 1614281322239677571L;

	private transient final SoapServerConfig config;

	private transient final ServerStats stats;

	private static final String HTML_TITLE = "<html><head><title>Welcome SOAP Binding Component</title></head><body>";

    /**
     * The probe counting unknown URLs received by the Jetty server.
     */
    private transient final CounterProbe probeInformationServlet;

    /**
     * The probe counting allocated threads in the thread pool of the HTTP
     * server.
     */
    private transient GaugeProbe<Long, Long> probeHttpServerThreadPoolAllocatedThreads;

    /**
     * The probe counting idle threads in the thread pool of the HTTP server.
     */
    private transient GaugeProbe<Long, Long> probeHttpServerThreadPoolIdleThreads;

    /**
     * The probe counting requests in the queue of the thread pool of the HTTP
     * server.
     */
    private transient GaugeProbe<Long, Long> probeHttpServerThreadPoolQueuedRequests;

    /**
     * The component logger
     */
    private transient final Logger logger;

    /**
     * 
     * @param config
     * @param stats
     * @param probes
     *            The technical monitoring probes
     * @param logger
     *            The component logger
     */
    public WelcomeServlet(final SoapServerConfig config, final ServerStats stats,
            final IncomingProbes probes, final Logger logger) {
		this.config = config;
		this.stats = stats;
        this.logger = logger;
        this.probeHttpServerThreadPoolAllocatedThreads = probes.probeHttpServerThreadPoolAllocatedThreads;
        this.probeHttpServerThreadPoolIdleThreads = probes.probeHttpServerThreadPoolIdleThreads;
        this.probeHttpServerThreadPoolQueuedRequests = probes.probeHttpServerThreadPoolQueuedRequests;
        this.probeInformationServlet = probes.probeInformationServlet;
	}

	/**
     * 
     */
	@Override
	protected void doGet(final HttpServletRequest req, final HttpServletResponse resp) throws ServletException, IOException {
		
        try {
            this.probeInformationServlet.inc();
            this.probeHttpServerThreadPoolAllocatedThreads.pick();
            this.probeHttpServerThreadPoolIdleThreads.pick();
            this.probeHttpServerThreadPoolQueuedRequests.pick();
        } catch (final ProbeNotStartedException e) {
            this.logger
                    .warning("HTTP probes are not started. Values of probes could be incorrect.");
        }

        // Redirection (forward) requested ? Proceed...
		String redirect = this.config.getRedirect(req.getRequestURI());
		if(redirect != null) {
			// Example: redirect="/petals/services/myWebService"
			if(! redirect.startsWith("/")) redirect = "/" + redirect;
			int pos = redirect.substring(1).indexOf("/"); // The 2nd '/' (in the example, before "services")
			String ctxname = redirect.substring(0, pos+1); // In the example: "/petals"
			ServletContext otherContext = getServletContext().getContext(ctxname);
			// In the example: forward to "/services/myWebService"
			otherContext.getRequestDispatcher(redirect.substring(pos+1)).forward(req, resp);
			return; // forward done !
		}
		
		// No (forward) redirection: display SOAP BC information.
		
		final ServletOutputStream out = resp.getOutputStream();
		out.write(HTML_TITLE.getBytes());
		out.write("<h1>Petals BC SOAP</h1>".getBytes());
		out.write("<h2>Component Configuration</h2>".getBytes());
		out.write("<ul>".getBytes());
        out.write(("<li>" + this.config.getHostToDisplay() + "</li>").getBytes());
		out.write(("<li>HTTP Port :                " + this.config.getHttpPort() + "</li>").getBytes());
		if(this.config.isHttpsEnabled()) {
		    out.write(("<li>HTTPS Port :                " + this.config.getHttpsPort() + "</li>").getBytes());
		}
		out.write(("<li>Jetty Acceptors :     " + this.config.getJettyAcceptors() + "</li>").getBytes());
		out.write(("<li>Jetty Max pool size : " + this.config.getJettyThreadMaxPoolSize() + "</li>").getBytes());
		out.write(("<li>Jetty Min pool size : " + this.config.getJettyThreadMinPoolSize() + "</li>").getBytes());
		out.write(("<li>Services Context :    " + this.config.getServicesContext() + "</li>").getBytes());
		out.write("</ul>".getBytes());

		out.write("<h2>Web Services information</h2>".getBytes());
		out.write("<ul>".getBytes());
		out.write("<li>Services List : ".getBytes());
		final String path = "/" + this.config.getServicesContext() + "/" + this.config.getServicesMapping() + "/" + SoapConstants.Component.MAPPING_NAME;
		final String link = "<a href='" + path + "'>" + path + "</a>";
		out.write(link.getBytes());
		out.write("</li>".getBytes());
		out.write("</ul>".getBytes());

		out.write("<h2>Server Stats</h2>".getBytes());
		out.write("<ul>".getBytes());
		out.write(("<li>Start time : " + new SimpleDateFormat().format(new Date(stats.getStartTime())) + "</li>").getBytes());
		out.write(("<li>Jetty Server version : " + Server.getVersion() + "</li>").getBytes());
		out.write(("<li>Incoming WS GET requests : " + stats.getGetRequests() + "</li>").getBytes());
		out.write(("<li>Incoming WS POST requests : " + stats.getPostRequests() + "</li>").getBytes());
		out.write("</ul>".getBytes());

		out.write("</body></html>".getBytes());

		out.flush();
		out.close();
	}
	
	@Override
	protected void doPost(final HttpServletRequest req, final HttpServletResponse resp) throws ServletException, IOException {
		doGet(req, resp);
	}
}
