/**
 * Copyright (c) 2007-2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.listener.outgoing;

import java.net.URI;

import org.ow2.petals.probes.api.key.ProbeKey;

/**
 * <p>
 * A probe key identifying a web-service client part:
 * <ol>
 * <li>URL of the external web-service,</li>
 * <li>invoked operation,</li>
 * <li>message exchange pattern used.</li>
 * </ol>
 * </p>
 * 
 * @author Christophe DENEUX (PetalsLink)
 * 
 */
public class ServiceClientKey implements ProbeKey {

	/**
	 * URL of the external web-service
	 */
	private final String address;

	/**
	 * Invoked operation
	 */
	private final String operation;

	/**
	 * MEP used
	 */
	private final URI mep;

	/**
	 * For optimization reason, the string representation of the
	 * ServiceClientKey is computed at constructor level as it is immutable, and
	 * not each time it is needed.
	 */
	private final String[] keyAsStringArray;

	/**
	 * Creates a new instance of ServiceClientKey
	 * 
	 * @param address
	 *            URL of the external web-service
	 * @param operation
	 *            Invoked operation
	 * @param mep
	 *            MEP used
	 */
	public ServiceClientKey(final String address, final String operation,
			final URI mep) {
		this.address = address;
		this.operation = operation;
		this.mep = mep;
		this.keyAsStringArray = new String[] { this.address, this.operation,
				this.mep.toString() };
	}

	// Generated by Eclipse
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((address == null) ? 0 : address.hashCode());
		result = prime * result + ((mep == null) ? 0 : mep.hashCode());
		result = prime * result
				+ ((operation == null) ? 0 : operation.hashCode());
		return result;
	}

	// Generated by Eclipse
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ServiceClientKey other = (ServiceClientKey) obj;
		if (address == null) {
			if (other.address != null)
				return false;
		} else if (!address.equals(other.address))
			return false;
		if (mep == null) {
			if (other.mep != null)
				return false;
		} else if (!mep.equals(other.mep))
			return false;
		if (operation == null) {
			if (other.operation != null)
				return false;
		} else if (!operation.equals(other.operation))
			return false;
		return true;
	}

	@Override
	public String[] toStringArray() {
		return this.keyAsStringArray;
	}

    @Override
    public String toReadableString() {
        return new StringBuilder(this.address).append(", ").append(this.mep.toASCIIString())
                .append(", ").append(this.operation).toString();
    }

    public String getAddress() {
        return this.address;
    }

    public String getOperation() {
        return operation;
    }

    public URI getMep() {
        return mep;
    }
}