/**
 * Copyright (c) 2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.monitoring;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Timer;

import javax.management.MBeanNotificationInfo;

import org.apache.commons.pool.impl.GenericObjectPool;
import org.mortbay.thread.BoundedThreadPool;
import org.ow2.petals.binding.soap.listener.incoming.jetty.IncomingProbes;
import org.ow2.petals.binding.soap.listener.incoming.jetty.IncomingServiceKey;
import org.ow2.petals.binding.soap.listener.outgoing.OutgoingProbes;
import org.ow2.petals.binding.soap.listener.outgoing.ServiceClientKey;
import org.ow2.petals.binding.soap.monitoring.defect.HTTPServerThreadPoolExhaustedDefectCreator;
import org.ow2.petals.binding.soap.monitoring.defect.ServiceClientPoolExhaustedDefectCreator;
import org.ow2.petals.binding.soap.monitoring.sensor.HttpThreadPoolAllocatedThreadsDefectDetector;
import org.ow2.petals.binding.soap.monitoring.sensor.HttpThreadPoolAllocatedThreadsGaugeSensor;
import org.ow2.petals.binding.soap.monitoring.sensor.HttpThreadPoolIdleThreadsGaugeSensor;
import org.ow2.petals.binding.soap.monitoring.sensor.HttpThreadPoolQueuedRequestsGaugeSensor;
import org.ow2.petals.binding.soap.monitoring.sensor.WsClientPoolExhaustionsDefectDetector;
import org.ow2.petals.binding.soap.monitoring.sensor.WsClientPoolsClientsInUseGaugeSensor;
import org.ow2.petals.probes.api.KeyedProbesFactory;
import org.ow2.petals.probes.api.KeyedProbesFactoryBuilder;
import org.ow2.petals.probes.api.ProbesFactory;
import org.ow2.petals.probes.api.ProbesFactoryBuilder;
import org.ow2.petals.probes.api.enums.ExecutionStatus;
import org.ow2.petals.probes.api.exceptions.MultipleProbesFactoriesFoundException;
import org.ow2.petals.probes.api.exceptions.NoProbesFactoryFoundException;
import org.ow2.petals.probes.api.exceptions.ProbeInitializationException;
import org.ow2.petals.probes.api.exceptions.ProbeInitializedException;
import org.ow2.petals.probes.api.exceptions.ProbeNotInitializedException;
import org.ow2.petals.probes.api.exceptions.ProbeNotStartedException;
import org.ow2.petals.probes.api.exceptions.ProbeShutdownException;
import org.ow2.petals.probes.api.exceptions.ProbeStartedException;
import org.ow2.petals.probes.api.exceptions.ProbeStartupException;
import org.ow2.petals.probes.api.exceptions.ProbeStopException;
import org.ow2.petals.probes.api.key.ExecutionStatusProbeKey;
import org.ow2.petals.probes.api.key.ProbeKey;
import org.ow2.petals.probes.api.probes.CounterProbe;
import org.ow2.petals.probes.api.probes.GaugeProbe;
import org.ow2.petals.probes.api.probes.KeyedCounterProbe;
import org.ow2.petals.probes.api.probes.KeyedCounterProbeWithExecutionStatus;
import org.ow2.petals.probes.api.probes.KeyedGaugeProbe;
import org.ow2.petals.probes.api.probes.KeyedResponseTimeProbe;
import org.ow2.petals.probes.api.probes.ResponseTimeRelativeValues;

/**
 * The monitoring MBean of the BC SOAP
 * 
 * @author Christophe DENEUX (PetalsLink)
 */
public class Monitoring extends org.ow2.petals.component.framework.monitoring.Monitoring implements
        MonitoringMBean {

    // --- Probes --- //

    /**
     * The probe counting unknown URLs received by the Jetty server.
     */
    private final CounterProbe probeUnknownServlet;

    /**
     * The probe counting unknown URLs received by the Jetty server.
     */
    private final CounterProbe probeInformationServlet;

    /**
     * The probe counting service invocation by web-service, web-service client
     * and execution status, received by the Jetty server.
     */
    private final KeyedCounterProbeWithExecutionStatus<ProbeKey> probeHttpRequestsInvocationsCount;

    /**
     * The probe measuring the response time of service invocation by
     * web-service, web-service client and execution status, received by the
     * Jetty server.
     */
    private final KeyedResponseTimeProbe<IncomingServiceKey> probeIncomingWsRequestResponseTime;

    /**
     * The probe measuring the response time of WS client invocation by external
     * web-service, operation, MEP and execution status.
     */
    private final KeyedResponseTimeProbe<ServiceClientKey> probeOutgoingWsRequestResponseTime;

    /**
     * The probe counting allocated threads in the thread pool of the HTTP
     * server.
     */
    private GaugeProbe<Long, Long> probeHttpServerThreadPoolAllocatedThreads;

    /**
     * The probe counting idle threads in the thread pool of the HTTP server.
     */
    private GaugeProbe<Long, Long> probeHttpServerThreadPoolIdleThreads;

    /**
     * The probe counting requests in the queue of the thread pool of the HTTP
     * server.
     */
    private GaugeProbe<Long, Long> probeHttpServerThreadPoolQueuedRequests;

    /**
     * The probe counting allocated ws-clients in the pool.
     */
    private KeyedGaugeProbe<ServiceClientKey, Long, Long> probeWsClientPoolClientsInUse;

    /**
     * The probe counting the number of ws-client pool exhaustions.
     */
    private KeyedCounterProbe<ProbeKey> probeWsClientPoolExhaustions;

    /**
     * The probe counting external web-service invocation by web-service,
     * web-service operation and MEP.
     */
    private final KeyedCounterProbeWithExecutionStatus<ProbeKey> probeWsRequestsInvocationsCount;

    /**
     * The thread pool of the HTTP server
     */
    private BoundedThreadPool httpThreadPool;

    /**
     * The pools of service clients used to call external web services.
     */
    private Map<ServiceClientKey, GenericObjectPool> wsClientPools;

    /**
     * Technical monitoring probes needed by incoming request processing
     */
    private final IncomingProbes incomingProbes;

    /**
     * Technical monitoring probes needed by outgoing request processing
     */
    private final OutgoingProbes outgoingProbes;

    /**
     * Sensor attached to the allocated threads of the HTTP thread pool.
     */
    private final HttpThreadPoolAllocatedThreadsGaugeSensor httpThreadPoolAllocatedThreadsGaugeSensor;

    /**
     * Defect detector attached to the allocated threads of the HTTP thread
     * pool.
     */
    private final HttpThreadPoolAllocatedThreadsDefectDetector httpThreadPoolAllocatedThreadsDefectDetector;

    /**
     * Defect creator attached to the defect 'HTTP thread pool exhausted'.
     */
    private final HTTPServerThreadPoolExhaustedDefectCreator httpServerThreadPoolExhaustedDefectCreator;

    /**
     * Sensor attached to the idle threads of the HTTP thread pool.
     */
    private final HttpThreadPoolIdleThreadsGaugeSensor httpThreadPoolIdleThreadsGaugeSensor;

    /**
     * Sensor attached to the queued requests of the HTTP thread pool.
     */
    private final HttpThreadPoolQueuedRequestsGaugeSensor httpThreadPoolQueuedRequestsGaugeSensor;

    /**
     * Sensor attached to the clients in use of the pools of service clients
     */
    private final WsClientPoolsClientsInUseGaugeSensor wsClientPoolsClientsInUseGaugeSensor;

    /**
     * Defect creator attached to the defect 'Web-service client pool
     * exhausted'.
     */
    private final ServiceClientPoolExhaustedDefectCreator serviceClientPoolExhaustedDefectCreator;

    /**
     * Creates the monitoring MBean
     * 
     * @param responseTimeProbesTimer
     *            The timer used as sampler by response time probes. <b>The
     *            caller is responsible to cancel the timer to free resources
     *            attached</b>.
     * @param samplePeriod
     *            The duration of a sample, in milliseconds.
     * @throws NoProbesFactoryFoundException
     *             No probe implementation found in the classloader
     * @throws MultipleProbesFactoriesFoundException
     *             Several probe implementations found in the classloader
     */
    public Monitoring(final Timer responseTimeProbesTimer, final long samplePeriod)
            throws MultipleProbesFactoriesFoundException, NoProbesFactoryFoundException {

        super(responseTimeProbesTimer, samplePeriod);

        final ProbesFactoryBuilder<Long, Long> probesFactoryBuilder = new ProbesFactoryBuilder<Long, Long>();
        final ProbesFactory<Long, Long> probesFactory = probesFactoryBuilder
                .getProbesFactory();

        this.probeUnknownServlet = probesFactory.createCounterProbe();
        this.probeInformationServlet = probesFactory.createCounterProbe();
        this.httpThreadPoolAllocatedThreadsGaugeSensor = new HttpThreadPoolAllocatedThreadsGaugeSensor();
        this.httpServerThreadPoolExhaustedDefectCreator = new HTTPServerThreadPoolExhaustedDefectCreator(
                this);
        this.httpThreadPoolAllocatedThreadsDefectDetector = new HttpThreadPoolAllocatedThreadsDefectDetector(
                new HTTPServerThreadPoolExhaustedDefectCreator(this));
        this.probeHttpServerThreadPoolAllocatedThreads = probesFactory.createGaugeProbe(
                this.httpThreadPoolAllocatedThreadsGaugeSensor,
                this.httpThreadPoolAllocatedThreadsDefectDetector);
        this.httpThreadPoolIdleThreadsGaugeSensor = new HttpThreadPoolIdleThreadsGaugeSensor();
        this.probeHttpServerThreadPoolIdleThreads = probesFactory
                .createGaugeProbe(this.httpThreadPoolIdleThreadsGaugeSensor);
        this.httpThreadPoolQueuedRequestsGaugeSensor = new HttpThreadPoolQueuedRequestsGaugeSensor();
        this.probeHttpServerThreadPoolQueuedRequests = probesFactory
                .createGaugeProbe(this.httpThreadPoolQueuedRequestsGaugeSensor);
        
        final KeyedProbesFactoryBuilder<IncomingServiceKey, Long, Long> incomingProbesFactoryBuilder = new KeyedProbesFactoryBuilder<IncomingServiceKey, Long, Long>();
        final KeyedProbesFactory<IncomingServiceKey, Long, Long> incomingProbesFactory = incomingProbesFactoryBuilder
                .getKeyedProbesFactory();

        this.probeHttpRequestsInvocationsCount = incomingProbesFactory
                .createKeyedCounterProbeWithExecutionStatus();
        this.probeIncomingWsRequestResponseTime = incomingProbesFactory
                .createKeyedResponseTimeProbe(responseTimeProbesTimer, samplePeriod);
        this.incomingProbes = new IncomingProbes(this.probeUnknownServlet,
                this.probeInformationServlet, this.probeHttpServerThreadPoolAllocatedThreads,
                this.probeHttpServerThreadPoolIdleThreads,
                this.probeHttpServerThreadPoolQueuedRequests,
                this.probeHttpRequestsInvocationsCount, this.probeIncomingWsRequestResponseTime);

        final KeyedProbesFactoryBuilder<ServiceClientKey, Long, Long> outgoingProbesFactoryBuilder = new KeyedProbesFactoryBuilder<ServiceClientKey, Long, Long>();
        final KeyedProbesFactory<ServiceClientKey, Long, Long> outgoingProbesFactory = outgoingProbesFactoryBuilder
                .getKeyedProbesFactory();

        this.probeOutgoingWsRequestResponseTime = outgoingProbesFactory
                .createKeyedResponseTimeProbe(responseTimeProbesTimer, samplePeriod);
        this.serviceClientPoolExhaustedDefectCreator = new ServiceClientPoolExhaustedDefectCreator(
                this);
        this.probeWsClientPoolExhaustions = outgoingProbesFactory
                .createKeyedCounterProbe(new WsClientPoolExhaustionsDefectDetector(
                        this.serviceClientPoolExhaustedDefectCreator));
        this.wsClientPoolsClientsInUseGaugeSensor = new WsClientPoolsClientsInUseGaugeSensor();
        this.probeWsClientPoolClientsInUse = outgoingProbesFactory
                .createKeyedGaugeProbe(this.wsClientPoolsClientsInUseGaugeSensor);
        this.probeWsRequestsInvocationsCount = outgoingProbesFactory
                .createKeyedCounterProbeWithExecutionStatus();

        this.outgoingProbes = new OutgoingProbes(this.probeWsClientPoolClientsInUse,
                this.probeWsClientPoolExhaustions, this.probeWsRequestsInvocationsCount,
                this.probeOutgoingWsRequestResponseTime);

    }

    @Override
    public void doInit()
            throws ProbeInitializedException,
            ProbeStartedException, ProbeInitializationException {

        this.probeUnknownServlet.init();
        this.probeInformationServlet.init();
        this.probeHttpServerThreadPoolAllocatedThreads.init();
        this.probeHttpServerThreadPoolIdleThreads.init();
        this.probeHttpServerThreadPoolQueuedRequests.init();
        this.probeHttpRequestsInvocationsCount.init();
        this.probeIncomingWsRequestResponseTime.init();
        this.probeWsClientPoolClientsInUse.init();
        this.probeWsClientPoolExhaustions.init();
        this.probeWsRequestsInvocationsCount.init();
        this.probeOutgoingWsRequestResponseTime.init();
    }

    public void setHttpThreadPool(final BoundedThreadPool httpThreadPool) {
        this.httpThreadPool = httpThreadPool;
    }

    public void setWsClientPools(final Map<ServiceClientKey, GenericObjectPool> wsClientPools) {
        this.wsClientPools = wsClientPools;
    }

    @Override
    public void doStart() throws ProbeNotInitializedException, ProbeStartedException,
            ProbeStartupException {
        this.httpThreadPoolAllocatedThreadsDefectDetector.setHttpThreadPool(this.httpThreadPool);
        this.httpThreadPoolAllocatedThreadsGaugeSensor.setHttpThreadPool(this.httpThreadPool);
        this.httpThreadPoolIdleThreadsGaugeSensor.setHttpThreadPool(this.httpThreadPool);
        this.httpThreadPoolQueuedRequestsGaugeSensor.setHttpThreadPool(this.httpThreadPool);

        this.wsClientPoolsClientsInUseGaugeSensor.setWsClientPools(this.wsClientPools);

        this.probeUnknownServlet.start();
        this.probeInformationServlet.start();
        this.probeHttpServerThreadPoolAllocatedThreads.start();
        this.probeHttpServerThreadPoolIdleThreads.start();
        this.probeHttpServerThreadPoolQueuedRequests.start();
        this.probeHttpRequestsInvocationsCount.start();
        this.probeIncomingWsRequestResponseTime.start();
        this.probeWsClientPoolClientsInUse.start();
        this.probeWsClientPoolExhaustions.start();
        this.probeWsRequestsInvocationsCount.start();
        this.probeOutgoingWsRequestResponseTime.start();
    }

    @Override
    public void doStop() throws ProbeNotInitializedException, ProbeNotStartedException,
            ProbeStopException {
        this.probeUnknownServlet.stop();
        this.probeInformationServlet.stop();
        this.probeHttpServerThreadPoolAllocatedThreads.stop();
        this.probeHttpServerThreadPoolIdleThreads.stop();
        this.probeHttpServerThreadPoolQueuedRequests.stop();
        this.probeHttpRequestsInvocationsCount.stop();
        this.probeIncomingWsRequestResponseTime.stop();
        this.probeWsClientPoolClientsInUse.stop();
        this.probeWsClientPoolExhaustions.stop();
        this.probeWsRequestsInvocationsCount.stop();
        this.probeOutgoingWsRequestResponseTime.stop();
    }

    @Override
    public void doShutdown() throws ProbeShutdownException, ProbeStartedException,
            ProbeNotInitializedException {
        this.probeUnknownServlet.shutdown();
        this.probeInformationServlet.shutdown();
        this.probeHttpServerThreadPoolAllocatedThreads.shutdown();
        this.probeHttpServerThreadPoolIdleThreads.shutdown();
        this.probeHttpServerThreadPoolQueuedRequests.shutdown();
        this.probeHttpRequestsInvocationsCount.shutdown();
        this.probeIncomingWsRequestResponseTime.shutdown();
        this.probeWsClientPoolClientsInUse.shutdown();
        this.probeWsClientPoolExhaustions.shutdown();
        this.probeWsRequestsInvocationsCount.shutdown();
        this.probeOutgoingWsRequestResponseTime.shutdown();
    }

    @Override
    public List<MBeanNotificationInfo> addNotificationInfo() {
        return Arrays.asList(new MBeanNotificationInfo[] {
                this.httpServerThreadPoolExhaustedDefectCreator.getNotificationInfo(),
                this.serviceClientPoolExhaustedDefectCreator.getNotificationInfo() });
    }

    // Probe accessors //

    /**
     * @return Technical monitoring probes needed by incoming request processing
     */
    public IncomingProbes getIncomingProbes() {
        return this.incomingProbes;
    }

    /**
     * @return Technical monitoring probes needed by outgoing request processing
     */
    public OutgoingProbes getOutgoingProbes() {
        return this.outgoingProbes;
    }

    // ---------------- //

    @Override
    public long getHttpServerThreadPoolMaxSize() {
        return this.httpThreadPool.getMaxThreads();
    }

    @Override
    public long getHttpServerThreadPoolMinSize() {
        return this.httpThreadPool.getMinThreads();
    }

    @Override
    public long getHttpServerThreadPoolAllocatedThreadsMax() throws ProbeNotInitializedException {
        return this.probeHttpServerThreadPoolAllocatedThreads.getMaxValue();
    }

    @Override
    public long getHttpServerThreadPoolAllocatedThreadsCurrent() throws ProbeNotStartedException {
        return this.probeHttpServerThreadPoolAllocatedThreads.getInstantValue();
    }

    @Override
    public long getHttpServerThreadPoolIdleThreadsMax() throws ProbeNotInitializedException {
        return this.probeHttpServerThreadPoolIdleThreads.getMaxValue();
    }

    @Override
    public long getHttpServerThreadPoolIdleThreadsCurrent() throws ProbeNotStartedException {
        return this.probeHttpServerThreadPoolIdleThreads.getInstantValue();
    }

    @Override
    public long getHttpServerThreadPoolQueuedRequestsMax() throws ProbeNotInitializedException {
        return this.probeHttpServerThreadPoolQueuedRequests.getMaxValue();
    }

    @Override
    public long getHttpServerThreadPoolQueuedRequestsCurrent() throws ProbeNotStartedException {
        return this.probeHttpServerThreadPoolQueuedRequests.getInstantValue();
    }

    // -------------------//

    @Override
    public long getUnknownURLsCounter() throws ProbeNotInitializedException {
        return this.probeUnknownServlet.getValue();
    }

    // -------------------//

    @Override
    public long getInformationURLsCounter() throws ProbeNotInitializedException {
        return this.probeInformationServlet.getValue();
    }

    // -------------------//

    @Override
    public Map<String[], Long> getWsClientPoolClientsInUseMax()
            throws ProbeNotInitializedException {
        return this.probeWsClientPoolClientsInUse.getConvertedMaxValues();
    }

    @Override
    public Map<String[], Long> getWsClientPoolClientsInUseCurrent()
            throws ProbeNotStartedException {
        return this.probeWsClientPoolClientsInUse.getConvertedInstantValues();
    }

    @Override
    public Map<String[], Long> getWsClientPoolExhaustions() throws ProbeNotInitializedException {
        return this.probeWsClientPoolExhaustions.getConvertedValues();
    }

    @Override
    public Map<String[], Long> getIncomingWsRequestsCounter() throws ProbeNotInitializedException {
        return this.probeHttpRequestsInvocationsCount.getConvertedValues();
    }

    @Override
    public Map<String, Long> getHttpRequestsCount(final String webService)
            throws ProbeNotInitializedException {
        final Map<String, Long> result = new HashMap<String, Long>(ExecutionStatus.values().length);
        result.put(ExecutionStatus.ERROR.toString(), Long.valueOf(0L));
        result.put(ExecutionStatus.FAULT.toString(), Long.valueOf(0L));
        result.put(ExecutionStatus.PENDING.toString(), Long.valueOf(0L));
        result.put(ExecutionStatus.SUCCEEDED.toString(), Long.valueOf(0L));
        if (webService != null && !webService.isEmpty()) {
            final Map<ExecutionStatusProbeKey<ProbeKey>, Long> requestInvocationsCount = this.probeHttpRequestsInvocationsCount
                    .getValues();
            final Iterator<Map.Entry<ExecutionStatusProbeKey<ProbeKey>, Long>> itRequestInvocationsCount = requestInvocationsCount
                    .entrySet().iterator();
            while (itRequestInvocationsCount.hasNext()) {
                final Map.Entry<ExecutionStatusProbeKey<ProbeKey>, Long> requestInvocationCount = itRequestInvocationsCount
                        .next();
                final ExecutionStatusProbeKey<ProbeKey> key = requestInvocationCount
                        .getKey();
                final IncomingServiceKey originalKey = (IncomingServiceKey) key.getOriginalKey();
                if (originalKey.getWebServicePath().equals(webService)) {
                    final String executionStatus = key.getExecutionStatus().toString();
                    final Long httpRequestsCount = result.get(executionStatus);
                    if (httpRequestsCount == null) {
                        result.put(executionStatus, requestInvocationCount.getValue());
                    } else {
                        result.put(executionStatus,
                                httpRequestsCount + requestInvocationCount.getValue());
                    }
                }
            }
        }
        return result;
    }

    @Override
    public Map<String, Long> getHttpRequestsCount(final String webService,
            final String webServiceClient) throws ProbeNotInitializedException {
        final Map<String, Long> result = new HashMap<String, Long>(ExecutionStatus.values().length);
        result.put(ExecutionStatus.ERROR.toString(), Long.valueOf(0L));
        result.put(ExecutionStatus.FAULT.toString(), Long.valueOf(0L));
        result.put(ExecutionStatus.PENDING.toString(), Long.valueOf(0L));
        result.put(ExecutionStatus.SUCCEEDED.toString(), Long.valueOf(0L));
        if (webService != null && !webService.isEmpty() && webServiceClient != null
                && !webServiceClient.isEmpty()) {
            final Map<ExecutionStatusProbeKey<ProbeKey>, Long> requestInvocationsCount = this.probeHttpRequestsInvocationsCount
                    .getValues();
            final Iterator<Map.Entry<ExecutionStatusProbeKey<ProbeKey>, Long>> itRequestInvocationsCount = requestInvocationsCount
                    .entrySet().iterator();
            while (itRequestInvocationsCount.hasNext()) {
                final Map.Entry<ExecutionStatusProbeKey<ProbeKey>, Long> requestInvocationCount = itRequestInvocationsCount
                        .next();
                final ExecutionStatusProbeKey<ProbeKey> key = requestInvocationCount
                        .getKey();
                final IncomingServiceKey originalKey = (IncomingServiceKey) key.getOriginalKey();
                // TODO: Gérer le fait que le web-service client peut-être une
                // adresse IP ou un nom DNS
                if (originalKey.getWebServicePath().equals(webService)
                        && originalKey.getWebServiceClient().equals(webServiceClient)) {
                    final String executionStatus = key.getExecutionStatus().toString();
                    final Long httpRequestsCount = result.get(executionStatus);
                    if (httpRequestsCount == null) {
                        result.put(executionStatus, requestInvocationCount.getValue());
                    } else {
                        result.put(executionStatus,
                                httpRequestsCount + requestInvocationCount.getValue());
                    }
                }
            }
        }
        return result;
    }

    @Override
    public Map<String[], Long[]> getIncomingWsRequestsResponseTimeAbs()
            throws ProbeNotInitializedException {
        return this.probeIncomingWsRequestResponseTime
                .getConvertedAbsoluteResponseTimeValues();
    }

    @Override
    public Map<String[], Long[]> getIncomingWsRequestsResponseTimeRel()
            throws ProbeNotStartedException {
        return this.probeIncomingWsRequestResponseTime
                .getConvertedRelativeResponseTimeValues();
    }

    @Override
    public Map<String[], Long[]> getIncomingWsRequestsResponseTimeRelGroupedBySvcAndExecStatus()
            throws ProbeNotStartedException {

        final Map<String[], Long[]> agregatedResults = new HashMap<String[], Long[]>();

        final Map<ExecutionStatusProbeKey<IncomingServiceKey>, ResponseTimeRelativeValues> responseTimeValues = this.probeIncomingWsRequestResponseTime
                .getRelativeResponseTimeValues();
        final Iterator<Map.Entry<ExecutionStatusProbeKey<IncomingServiceKey>, ResponseTimeRelativeValues>> itResponseTimeValues = responseTimeValues
                .entrySet().iterator();
        while (itResponseTimeValues.hasNext()) {
            final Map.Entry<ExecutionStatusProbeKey<IncomingServiceKey>, ResponseTimeRelativeValues> responseTimeValue = itResponseTimeValues
                    .next();
            final ExecutionStatusProbeKey<IncomingServiceKey> key = responseTimeValue.getKey();
            final IncomingServiceKey originalKey = key.getOriginalKey();
            final String[] newKey = new String[] { originalKey.getWebServicePath(),
                    originalKey.getWebServiceOperation(),
                    key.getExecutionStatus().toString() };
            if (agregatedResults.containsKey(newKey)) {
                // A new ws-client @IP found --> Aggregation
                final Long[] newValues = agregatedResults.get(newKey);
                final ResponseTimeRelativeValues values = responseTimeValue.getValue();
                newValues[0] = Math.max(values.getMax(), newValues[0]);
                newValues[1] = Math.min(values.getMin(), newValues[1]);
                newValues[2] += values.getResponseTimesSum();
                newValues[3] += values.getResponseTimesNb();
                // TODO: How to aggregate 2 values of percentile ?
                newValues[3] = values.getPercent10();
                newValues[4] = values.getPercent50();
                newValues[5] = values.getPercent90();
                agregatedResults.put(newKey, newValues);

            } else {
                final ResponseTimeRelativeValues values = responseTimeValue.getValue();
                final Long[] newValues = new Long[7];
                newValues[0] = values.getMax();
                newValues[1] = values.getMin();
                newValues[2] = values.getResponseTimesSum();
                newValues[3] = values.getResponseTimesNb();
                newValues[4] = values.getPercent10();
                newValues[5] = values.getPercent50();
                newValues[6] = values.getPercent90();
                agregatedResults.put(newKey, newValues);
            }
        }

        final Map<String[], Long[]> results = new HashMap<String[], Long[]>();
        for (final Map.Entry<String[], Long[]> entry : agregatedResults.entrySet()) {
            final Long[] values = entry.getValue();
            final Long[] newValues = new Long[6];
            newValues[0] = values[0];
            newValues[1] = values[2] / values[3];
            newValues[2] = values[1];
            newValues[3] = values[4];
            newValues[4] = values[5];
            newValues[5] = values[6];

            results.put(entry.getKey(), newValues);
        }

        return results;
    }

    @Override
    public Map<String, Long> getWsClientPool(String webService) throws ProbeNotStartedException,
            ProbeNotInitializedException {
        long currentInUse = 0L;
        long maxInUse = 0L;
        long exhaustions = 0L;
        long max = 0L;
        if (webService != null && !webService.isEmpty()) {
            final KeyedGaugeProbe<ServiceClientKey, Long, Long> clientPoolInUse = this.probeWsClientPoolClientsInUse;
            {
                final Iterator<Entry<ServiceClientKey, Long>> itClientPoolInUseCurrent = clientPoolInUse
                        .getInstantValues().entrySet().iterator();
                while (itClientPoolInUseCurrent.hasNext()) {
                    final Entry<ServiceClientKey, Long> clientPoolInUseCurrentEntry = itClientPoolInUseCurrent
                            .next();
                    final ServiceClientKey key = clientPoolInUseCurrentEntry
                            .getKey();
                    if (key.getAddress().equals(webService)) {
                        currentInUse += clientPoolInUseCurrentEntry.getValue();
                    }
                }
            }
            {
                final Iterator<Entry<ServiceClientKey, Long>> itClientPoolInUseMax = clientPoolInUse
                        .getMaxValues().entrySet().iterator();
                while (itClientPoolInUseMax.hasNext()) {
                    final Entry<ServiceClientKey, Long> clientPoolInUseMaxEntry = itClientPoolInUseMax
                            .next();
                    final ServiceClientKey key = clientPoolInUseMaxEntry
                            .getKey();
                    if (key.getAddress().equals(webService)) {
                        maxInUse += clientPoolInUseMaxEntry.getValue();
                    }
                }
            }
            final KeyedCounterProbe<ProbeKey> clientPoolExhaustion = this.probeWsClientPoolExhaustions;
            {
                final Iterator<Entry<ProbeKey, Long>> itClientPoolExhaustion = clientPoolExhaustion
                        .getValues().entrySet().iterator();
                while (itClientPoolExhaustion.hasNext()) {
                    final Entry<ProbeKey, Long> clientPoolInExhaustionEntry = itClientPoolExhaustion
                            .next();
                    final ServiceClientKey key = (ServiceClientKey) clientPoolInExhaustionEntry
                            .getKey();
                    if (key.getAddress().equals(webService)) {
                        exhaustions += clientPoolInExhaustionEntry.getValue();
                    }
                }
            }
            {
                final Iterator<Entry<ServiceClientKey, GenericObjectPool>> itClientPool = this.wsClientPools
                        .entrySet().iterator();
                while (itClientPool.hasNext()) {
                    final Entry<ServiceClientKey, GenericObjectPool> clientPoolEntry = itClientPool
                            .next();
                    final ServiceClientKey key = clientPoolEntry.getKey();
                    if (key.getAddress().equals(webService)) {
                        max += clientPoolEntry.getValue().getMaxActive();
                    }
                }
            }
        }

        final Map<String, Long> result = new HashMap<String, Long>(3);
        result.put("IN-USE-CURRENT", currentInUse);
        result.put("IN-USE-MAX", maxInUse);
        result.put("EXHAUSTIONS", exhaustions);
        result.put("MAX", max);
        return result;
    }

    @Override
    public Map<String[], Long> getOutgoingWsRequestsCounter() throws ProbeNotInitializedException {
        return this.probeWsRequestsInvocationsCount.getConvertedValues();
    }

    @Override
    public Map<String[], Long[]> getOutgoingWsRequestsResponseTimeAbs()
            throws ProbeNotInitializedException {
        return this.probeIncomingWsRequestResponseTime
                .getConvertedAbsoluteResponseTimeValues();
    }

    @Override
    public Map<String[], Long[]> getOutgoingWsRequestsResponseTimeRel()
            throws ProbeNotStartedException {
        return this.probeOutgoingWsRequestResponseTime
                .getConvertedRelativeResponseTimeValues();
    }

    @Override
    public Map<String[], Long[]> getOutgoingWsRequestsResponseTimeRelGroupedBySvcAndExecStatus()
            throws ProbeNotStartedException {

        final Map<String[], Long[]> agregatedResults = new HashMap<String[], Long[]>();

        final Map<ExecutionStatusProbeKey<ServiceClientKey>, ResponseTimeRelativeValues> responseTimeValues = this.probeOutgoingWsRequestResponseTime
                .getRelativeResponseTimeValues();
        final Iterator<Map.Entry<ExecutionStatusProbeKey<ServiceClientKey>, ResponseTimeRelativeValues>> itResponseTimeValues = responseTimeValues
                .entrySet().iterator();
        while (itResponseTimeValues.hasNext()) {
            final Map.Entry<ExecutionStatusProbeKey<ServiceClientKey>, ResponseTimeRelativeValues> responseTimeValue = itResponseTimeValues
                    .next();
            final ExecutionStatusProbeKey<ServiceClientKey> key = responseTimeValue.getKey();
            final ServiceClientKey originalKey = key.getOriginalKey();
            final String[] newKey = new String[] { originalKey.getAddress(),
                    originalKey.getOperation(), key.getExecutionStatus().toString() };
            if (agregatedResults.containsKey(newKey)) {
                // A new ws-client @IP found --> Aggregation
                final Long[] newValues = agregatedResults.get(newKey);
                final ResponseTimeRelativeValues values = responseTimeValue.getValue();
                newValues[0] = Math.max(values.getMax(), newValues[0]);
                newValues[1] = Math.min(values.getMin(), newValues[1]);
                newValues[2] += values.getResponseTimesSum();
                newValues[3] += values.getResponseTimesNb();
                // TODO: How to aggregate 2 values of percentile ?
                newValues[3] = values.getPercent10();
                newValues[4] = values.getPercent50();
                newValues[5] = values.getPercent90();
                agregatedResults.put(newKey, newValues);

            } else {
                final ResponseTimeRelativeValues values = responseTimeValue.getValue();
                final Long[] newValues = new Long[7];
                newValues[0] = values.getMax();
                newValues[1] = values.getMin();
                newValues[2] = values.getResponseTimesSum();
                newValues[3] = values.getResponseTimesNb();
                newValues[4] = values.getPercent10();
                newValues[5] = values.getPercent50();
                newValues[6] = values.getPercent90();
                agregatedResults.put(newKey, newValues);
            }
        }

        final Map<String[], Long[]> results = new HashMap<String[], Long[]>();
        for (final Map.Entry<String[], Long[]> entry : agregatedResults.entrySet()) {
            final Long[] values = entry.getValue();
            final Long[] newValues = new Long[6];
            newValues[0] = values[0];
            newValues[1] = values[2] / values[3];
            newValues[2] = values[1];
            newValues[3] = values[4];
            newValues[4] = values[5];
            newValues[5] = values[6];

            results.put(entry.getKey(), newValues);
        }

        return results;
    }

}
