/**
 * Copyright (c) 2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap.monitoring;

import java.util.Map;

import org.ow2.petals.probes.api.enums.ExecutionStatus;
import org.ow2.petals.probes.api.exceptions.ProbeNotInitializedException;
import org.ow2.petals.probes.api.exceptions.ProbeNotStartedException;

public interface MonitoringMBean extends
        org.ow2.petals.component.framework.monitoring.MonitoringMBean {

    public long getHttpServerThreadPoolMaxSize();

    public long getHttpServerThreadPoolMinSize();

    public long getHttpServerThreadPoolAllocatedThreadsMax() throws ProbeNotInitializedException;

    public long getHttpServerThreadPoolAllocatedThreadsCurrent() throws ProbeNotStartedException;

    public long getHttpServerThreadPoolIdleThreadsMax() throws ProbeNotInitializedException;

    public long getHttpServerThreadPoolIdleThreadsCurrent() throws ProbeNotStartedException;

    public long getHttpServerThreadPoolQueuedRequestsMax() throws ProbeNotInitializedException;

    public long getHttpServerThreadPoolQueuedRequestsCurrent() throws ProbeNotStartedException;

    public long getUnknownURLsCounter() throws ProbeNotInitializedException;

    public long getInformationURLsCounter() throws ProbeNotInitializedException;

    public Map<String[], Long> getWsClientPoolClientsInUseMax()
            throws ProbeNotInitializedException;

    public Map<String[], Long> getWsClientPoolClientsInUseCurrent()
            throws ProbeNotStartedException;

    public Map<String[], Long> getWsClientPoolExhaustions() throws ProbeNotInitializedException;

    /**
     * Returns the number of ws-requests since the last start. The map key is
     * structured as a {@link String} array where:
     * <ul>
     * <li>1st: the path (URL without scheme, server and port) of the invoked
     * web-service,</li>
     * <li>2nd: the web-service operation invoked,</li>
     * <li>3rd: the IP address of the web-service client,</li>
     * <li>4th: the execution status (see {@link ExecutionStatus} ).</li>
     * </ul>
     * Each value of the map entry is the number of requests associated to the
     * map entry key.
     * 
     * @return The number of requests for all web-services.
     * @throws ProbeNotStartedException
     */
    public Map<String[], Long> getIncomingWsRequestsCounter() throws ProbeNotInitializedException;

    /**
     * The number of requests for each execution status of the specified
     * web-service <code>webService</code> whatever the web-service client.
     * 
     * @param webService
     *            The web-service for which the number of requests for each
     *            execution status must be returned. Not null.
     * @return The number of requests for each execution status of the specified
     *         web-service <code>webService</code>. All execution status are
     *         contained in the map, even if set to 0. If
     *         <code>webService</code> is null, an empty map is returned.
     * @throws ProbeNotStartedException
     */
    public Map<String, Long> getHttpRequestsCount(final String webService)
            throws ProbeNotInitializedException;

    /**
     * The number of requests for each execution status of the specified
     * web-service <code>webService</code> and specified web-service-client
     * <code>webServiceClient</code>.
     * 
     * @param webService
     *            The web-service for which the number of requests for each
     *            execution status must be returned. Not null.
     * @param webServiceClient
     *            The IP address of the web-service client for which the number
     *            of requests for each execution status must be returned. Not
     *            null.
     * @return The number of requests for each execution status of the specified
     *         web-service <code>webService</code> and web-service client
     *         <code>webServiceClient</code>. All execution status are contained
     *         in the map, even if set to 0. If <code>webService</code> or
     *         <code>webServiceClient</code> is null, an empty map is returned.
     * @throws ProbeNotStartedException
     */
    public Map<String, Long> getHttpRequestsCount(final String webService,
            final String webServiceClient)
            throws ProbeNotInitializedException;

    /**
     * <p>
     * Returns the response time values of incoming ws-requests since the last
     * start.
     * </p>
     * <p>
     * The map key is structured as a {@link String} array where:
     * <ul>
     * <li>1st: the path (URL without scheme, server and port) of the invoked
     * web-service,</li>
     * <li>2nd: the web-service operation invoked,</li>
     * <li>3rd: the IP address of the web-service client,</li>
     * <li>4th: the execution status (see {@link ExecutionStatus} ).</li>
     * </ul>
     * Each value of the map entry is an array of response times where:
     * <ul>
     * <li>the first value is the max response time,</li>
     * <li>the second value is the average,</li>
     * <li>the third value is the min value.</li>
     * </ul>
     * </p>
     * 
     * @return The absolute response time values.
     * @throws ProbeNotStartedException
     */
    public Map<String[], Long[]> getIncomingWsRequestsResponseTimeAbs()
            throws ProbeNotInitializedException;

    /**
     * <p>
     * Returns the response time values of incoming ws-requests on a time
     * sample.
     * </p>
     * <p>
     * The map key is an array containing:
     * <ul>
     * <li>1st: the path (URL without scheme, server and port) of the invoked
     * web-service,</li>
     * <li>2nd: the web-service operation invoked,</li>
     * <li>3rd: the IP address of the web-service client,</li>
     * <li>4th: the execution status (see {@link ExecutionStatus} ).</li>
     * </ul>
     * </p>
     * <p>
     * Each value of the map entry is an array of response times where:
     * <ul>
     * <li>the first value is the max response time on the time sample,</li>
     * <li>the second value is the average,</li>
     * <li>the third value is the min value,</li>
     * <li>the fourth value is the 10-percentile,</li>
     * <li>the fifth value is the 50-percentile,</li>
     * <li>the sixth value is the 90-percentile.</li>
     * </ul>
     * </p>
     * 
     * @return The response time values on a time sample.
     * @throws ProbeNotStartedException
     */
    public Map<String[], Long[]> getIncomingWsRequestsResponseTimeRel()
            throws ProbeNotStartedException;

    /**
     * <p>
     * Returns the response time values of incoming ws-requests on a time sample
     * group by ws-path, ws-operation and execution status
     * </p>
     * <p>
     * The map key is an array containing:
     * <ul>
     * <li>1st: the path of the web-service (URL without scheme, server and
     * port),</li>
     * <li>2nd: the ws-operation,</li>
     * <li>3rd: execution status (SUCCEEDED, PENDING, FAULT, ERROR)</li>
     * </ul>
     * </p>
     * <p>
     * Each value of the map entry is an array of response times where:
     * <ul>
     * <li>the first value is the max response time on the time sample,</li>
     * <li>the second value is the average,</li>
     * <li>the third value is the min value,</li>
     * <li>the fourth value is the 10-percentile,</li>
     * <li>the fifth value is the 50-percentile,</li>
     * <li>the sixth value is the 90-percentile.</li>
     * </ul>
     * </p>
     * 
     * @return The response time values. All execution status are contained in
     *         the map, even if set to 0.
     * 
     * @throws ProbeNotStartedException
     */
    public Map<String[], Long[]> getIncomingWsRequestsResponseTimeRelGroupedBySvcAndExecStatus()
            throws ProbeNotStartedException;

    public Map<String, Long> getWsClientPool(final String webService)
            throws ProbeNotStartedException, ProbeNotInitializedException;

    /**
     * Returns the number of outgoing ws-requests since the last start. The map
     * key is structured as a {@link String} array where:
     * <ul>
     * <
     * <li>1st: the URL of the invoked web-service,</li>
     * <li>2nd: the invoked operation,</li>
     * <li>3rd: the MEP used,</li>
     * <li>4th: the execution status (see {@link ExecutionStatus} ) of the
     * requests.</li>
     * </ul>
     * Each value of the map entry is the number of requests associated to the
     * map entry key.
     * 
     * @return The number of requests for all web-services.
     * @throws ProbeNotStartedException
     */
    public Map<String[], Long> getOutgoingWsRequestsCounter() throws ProbeNotInitializedException;

    /**
     * <p>
     * Returns the response time values of outgoing ws-requests since the last
     * start.
     * </p>
     * <p>
     * The map key is structured as a {@link String} array where:
     * <ul>
     * <li>1st: the URL of the invoked web-service,</li>
     * <li>2nd: the invoked operation,</li>
     * <li>3rd: the MEP used,</li>
     * <li>4th: the execution status (see {@link ExecutionStatus} ) of the
     * requests.</li>
     * </ul>
     * Each value of the map entry is an array of response times where:
     * <ul>
     * <li>the first value is the max response time on the time sample,</li>
     * <li>the second value is the average,</li>
     * <li>the third value is the min value.</li>
     * </ul>
     * </p>
     * 
     * @return The absolute response time values.
     * @throws ProbeNotStartedException
     */
    public Map<String[], Long[]> getOutgoingWsRequestsResponseTimeAbs()
            throws ProbeNotInitializedException;

    /**
     * <p>
     * Returns the response time values of outgoing ws-requests on a time
     * sample.
     * </p>
     * <p>
     * The map key is an array containing:
     * <ul>
     * <li>1st: the URL of the external web-service,</li>
     * <li>2nd: the ws-operation invoked,</li>
     * <li>3rd: the mep with which the ws-operation is invoked,</li>
     * <li>4th: execution status (SUCCEEDED, PENDING, FAULT, ERROR)</li>
     * </ul>
     * </p>
     * <p>
     * Each value of the map entry is an array of response times where:
     * <ul>
     * <li>the first value is the max response time on the time sample,</li>
     * <li>the second value is the average,</li>
     * <li>the third value is the min value,</li>
     * <li>the fourth value is the 10-percentile,</li>
     * <li>the fifth value is the 50-percentile,</li>
     * <li>the sixth value is the 90-percentile.</li>
     * </ul>
     * </p>
     * 
     * @return The response time values on a time sample.
     * @throws ProbeNotStartedException
     */
    public Map<String[], Long[]> getOutgoingWsRequestsResponseTimeRel()
            throws ProbeNotStartedException;

    /**
     * <p>
     * Returns the response time values of outgoing ws-requests on a time sample
     * group by external web-service URL, ws-operation and execution status.
     * </p>
     * <p>
     * The map key is an array containing:
     * <ul>
     * <li>1st: the URL of the external web-service,</li>
     * <li>2nd: the ws-operation invoked,</li>
     * <li>4th: execution status (SUCCEEDED, PENDING, FAULT, ERROR)</li>
     * </ul>
     * </p>
     * <p>
     * Each value of the map entry is an array of response times where:
     * <ul>
     * <li>the first value is the max response time on the time sample,</li>
     * <li>the second value is the average,</li>
     * <li>the third value is the min value,</li>
     * <li>the fourth value is the 10-percentile,</li>
     * <li>the fifth value is the 50-percentile,</li>
     * <li>the sixth value is the 90-percentile.</li>
     * </ul>
     * </p>
     * 
     * @return The response time values. All execution status are contained in
     *         the map, even if set to 0.
     * 
     * @throws ProbeNotStartedException
     */
    public Map<String[], Long[]> getOutgoingWsRequestsResponseTimeRelGroupedBySvcAndExecStatus()
            throws ProbeNotStartedException;

}
