/**
 * Copyright (c) 2011-2012 EBM WebSourcing, 2012-2015 Linagora
 * 
 * This program/library is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This program/library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program/library; If not, see <http://www.gnu.org/licenses/>
 * for the GNU Lesser General Public License version 2.1.
 */
package org.ow2.petals.binding.soap;

import java.util.ArrayList;
import java.util.List;
import java.util.Timer;
import java.util.logging.Logger;

import javax.jbi.messaging.MessagingException;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;

import junit.framework.TestCase;

import org.ow2.petals.binding.soap.exception.ServiceClientPoolExhaustedException;
import org.ow2.petals.binding.soap.listener.outgoing.PetalsServiceClient;
import org.ow2.petals.binding.soap.monitoring.Monitoring;
import org.ow2.petals.component.framework.api.Message.MEPConstants;
import org.ow2.petals.component.framework.api.configuration.ConfigurationExtensions;
import org.ow2.petals.component.framework.jbidescriptor.generated.Component;
import org.ow2.petals.component.framework.jbidescriptor.generated.Provides;
import org.ow2.petals.component.framework.jbidescriptor.generated.Runtimepositivestrictint;
import org.ow2.petals.probes.api.exceptions.ProbeException;
import org.ow2.petals.probes.api.exceptions.ProbeInitializationException;
import org.ow2.petals.probes.api.exceptions.ProbeInitializedException;
import org.ow2.petals.probes.api.exceptions.ProbeNotInitializedException;
import org.ow2.petals.probes.api.exceptions.ProbeStartedException;
import org.ow2.petals.probes.api.exceptions.ProbeStartupException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.ebmwebsourcing.easycommons.xml.DocumentBuilders;

public class SoapComponentContextTest extends TestCase {
    
    private final static Logger LOG = Logger.getLogger(SoapComponentContextTest.class.getName());
    
    private final static int ACCEPTOR_POOL_SIZE = 1;
    
    private final static int PROCESSOR_MAX_POOL_SIZE = 5;
    
    private final static int PROCESSOR_CORE_POOL_SIZE = 2;

    private final Monitoring monitoringMbean;

    {
        try {
            this.monitoringMbean = new Monitoring(new Timer("Monitoring sampler", true), 300000);
        } catch (final ProbeException e) {
            e.printStackTrace();
            throw new AssertionError(e);
        }
    }

    /**
     * <p>
     * Check the max size of the web-service client pool if parameter
     * 'ws-clients-pool-size-max' is unset:
     * <ul>
     * <li>it is equals to the max size of the processor pool,</li>
     * <li>
     * check that the right exception is thrown when the pool is exhausted,</li>
     * <li>web-service client is available after one was returned</li>
     * </ul>
     * </p>
     * 
     */
    public void testBorrowServiceClient_002() throws MessagingException, ProbeInitializedException,
            ProbeStartedException, ProbeInitializationException, ProbeNotInitializedException,
            ProbeStartupException {
        
        final Component componentCfg = new Component();

        final Runtimepositivestrictint acceptorPoolSize = new Runtimepositivestrictint();
        acceptorPoolSize.setValue(ACCEPTOR_POOL_SIZE);
        componentCfg.setAcceptorPoolSize(acceptorPoolSize);
        
        final Runtimepositivestrictint processorMaxPoolSize = new Runtimepositivestrictint();
        processorMaxPoolSize.setValue(PROCESSOR_MAX_POOL_SIZE);
        componentCfg.setProcessorMaxPoolSize(processorMaxPoolSize);
        
        final Runtimepositivestrictint processorCorePoolSize = new Runtimepositivestrictint();
        processorCorePoolSize.setValue(PROCESSOR_CORE_POOL_SIZE);
        componentCfg.setProcessorPoolSize(processorCorePoolSize);
        
        final SoapComponentContext context = new SoapComponentContext(componentCfg,
                this.monitoringMbean.getOutgoingProbes(), LOG);
        final Provides provides = new Provides();
        provides.setEndpointName("MyEndpoint");
        provides.setTimeout(1000l);
        final ServiceContext<Provides> serviceCtx = context.getProvidersManager()
                .createServiceContext(provides);
        serviceCtx.setClassloader(Thread.currentThread().getContextClassLoader());
        
        this.monitoringMbean.init();
        this.monitoringMbean.setHttpThreadPool(null);
        this.monitoringMbean.setWsClientPools(context.getServiceClientPools());
        this.monitoringMbean.start();
        final List<PetalsServiceClient> serviceClients = new ArrayList<PetalsServiceClient>();
        try {
            for (int i = 0; i < PROCESSOR_MAX_POOL_SIZE; i++) {
                final PetalsServiceClient psc = context.borrowServiceClient("MyAdress", new QName(
                        "MyOperation"), "MySoapAction",
                        MEPConstants.IN_OUT_PATTERN.value(), new ConfigurationExtensions(null),
                        provides);
                TestCase.assertNotNull("The web-service client is null", psc);
                serviceClients.add(psc);
            }
        } catch (final ServiceClientPoolExhaustedException e) {
            TestCase.fail("The pool of web-service can't be increased to the number of message exchange processors.");
        }

        try {
            context.borrowServiceClient("MyAdress", new QName("MyOperation"), "MySoapAction",
                    MEPConstants.IN_OUT_PATTERN.value(), new ConfigurationExtensions(null),
                    provides);
            TestCase.fail("The pool of web-service is bigger than the message exchange processor.");
        } catch (final ServiceClientPoolExhaustedException e) {
            // Expected exception: no more web-service client is available, and
            // the pool can not be extended.
        }

        context.returnServiceClient("MyAdress", new QName("MyOperation"),
                MEPConstants.IN_OUT_PATTERN.value(), serviceClients.remove(0), "MySoapAction");

        try {
            final PetalsServiceClient psc = context.borrowServiceClient("MyAdress", new QName(
                    "MyOperation"), "MySoapAction", MEPConstants.IN_OUT_PATTERN.value(),
                    new ConfigurationExtensions(null), provides);
            TestCase.assertNotNull("The web-service client is null", psc);
            serviceClients.add(psc);
        } catch (final ServiceClientPoolExhaustedException e) {
            TestCase.fail("The pool of web-service is exhausted.");
        }
    }

    /**
     * <p>
     * Check the value returns by
     * {@link SoapComponentContext#getWsClientMaxPoolSize()} if parameter
     * 'ws-clients-pool-size-max' is set to a valid value.
     * </p>
     * <p>
     * Expected value is its default value: max size of the processor pool.
     * </p>
     * 
     * @throws MessagingException
     */
    public void testGetWsClientMaxPoolSize_000() throws MessagingException {

        final int wsClientMaxPoolSize = 23;

        final Component componentCfg = new Component();

        final Runtimepositivestrictint acceptorPoolSize = new Runtimepositivestrictint();
        acceptorPoolSize.setValue(ACCEPTOR_POOL_SIZE);
        componentCfg.setAcceptorPoolSize(acceptorPoolSize);

        final Runtimepositivestrictint processorMaxPoolSize = new Runtimepositivestrictint();
        processorMaxPoolSize.setValue(PROCESSOR_MAX_POOL_SIZE);
        componentCfg.setProcessorMaxPoolSize(processorMaxPoolSize);

        final Runtimepositivestrictint processorCorePoolSize = new Runtimepositivestrictint();
        processorCorePoolSize.setValue(PROCESSOR_CORE_POOL_SIZE);
        componentCfg.setProcessorPoolSize(processorCorePoolSize);

        final DocumentBuilder docBuilder = DocumentBuilders.takeDocumentBuilder();
        final Document doc = docBuilder.newDocument();
        final Element wsClientPoolSizeMaxElt = doc.createElementNS(
                "http://petals.ow2.org/components/soap/version-4", "ws-clients-pool-size-max");
        wsClientPoolSizeMaxElt.setTextContent(String.valueOf(wsClientMaxPoolSize));
        componentCfg.getAny().add(wsClientPoolSizeMaxElt);

        final SoapComponentContext context = new SoapComponentContext(componentCfg,
                this.monitoringMbean.getOutgoingProbes(), LOG);

        TestCase.assertEquals("The configured value is not used", wsClientMaxPoolSize,
                context.getWsClientMaxPoolSize());
    }

    /**
     * <p>
     * Check the value returns by
     * {@link SoapComponentContext#getWsClientMaxPoolSize()} if parameter
     * 'ws-clients-pool-size-max' is negative.
     * </p>
     * <p>
     * Expected value is its default value: max size of the processor pool.
     * </p>
     * 
     * @throws MessagingException
     */
    public void testGetWsClientMaxPoolSize_100() throws MessagingException {

        final Component componentCfg = new Component();

        final Runtimepositivestrictint acceptorPoolSize = new Runtimepositivestrictint();
        acceptorPoolSize.setValue(ACCEPTOR_POOL_SIZE);
        componentCfg.setAcceptorPoolSize(acceptorPoolSize);

        final Runtimepositivestrictint processorMaxPoolSize = new Runtimepositivestrictint();
        processorMaxPoolSize.setValue(PROCESSOR_MAX_POOL_SIZE);
        componentCfg.setProcessorMaxPoolSize(processorMaxPoolSize);

        final Runtimepositivestrictint processorCorePoolSize = new Runtimepositivestrictint();
        processorCorePoolSize.setValue(PROCESSOR_CORE_POOL_SIZE);
        componentCfg.setProcessorPoolSize(processorCorePoolSize);

        final DocumentBuilder docBuilder = DocumentBuilders.takeDocumentBuilder();
        final Document doc = docBuilder.newDocument();
        final Element wsClientPoolSizeMaxElt = doc.createElementNS(
                "http://petals.ow2.org/components/soap/version-4", "ws-clients-pool-size-max");
        wsClientPoolSizeMaxElt.setTextContent("-15");
        componentCfg.getAny().add(wsClientPoolSizeMaxElt);

        final SoapComponentContext context = new SoapComponentContext(componentCfg,
                this.monitoringMbean.getOutgoingProbes(), LOG);

        TestCase.assertEquals("Default value is not returned", PROCESSOR_MAX_POOL_SIZE,
                context.getWsClientMaxPoolSize());
    }

    /**
     * <p>
     * Check the value returns by
     * {@link SoapComponentContext#getWsClientMaxPoolSize()} if parameter
     * 'ws-clients-pool-size-max' is set to 0.
     * </p>
     * <p>
     * Expected value is its default value: max size of the processor pool.
     * </p>
     * 
     * @throws MessagingException
     */
    public void testGetWsClientMaxPoolSize_101() throws MessagingException {

        final Component componentCfg = new Component();

        final Runtimepositivestrictint acceptorPoolSize = new Runtimepositivestrictint();
        acceptorPoolSize.setValue(ACCEPTOR_POOL_SIZE);
        componentCfg.setAcceptorPoolSize(acceptorPoolSize);

        final Runtimepositivestrictint processorMaxPoolSize = new Runtimepositivestrictint();
        processorMaxPoolSize.setValue(PROCESSOR_MAX_POOL_SIZE);
        componentCfg.setProcessorMaxPoolSize(processorMaxPoolSize);

        final Runtimepositivestrictint processorCorePoolSize = new Runtimepositivestrictint();
        processorCorePoolSize.setValue(PROCESSOR_CORE_POOL_SIZE);
        componentCfg.setProcessorPoolSize(processorCorePoolSize);

        final DocumentBuilder docBuilder = DocumentBuilders.takeDocumentBuilder();
        final Document doc = docBuilder.newDocument();
        final Element wsClientPoolSizeMaxElt = doc.createElementNS(
                "http://petals.ow2.org/components/soap/version-4", "ws-clients-pool-size-max");
        wsClientPoolSizeMaxElt.setTextContent("0");
        componentCfg.getAny().add(wsClientPoolSizeMaxElt);

        final SoapComponentContext context = new SoapComponentContext(componentCfg,
                this.monitoringMbean.getOutgoingProbes(), LOG);

        TestCase.assertEquals("Default value is not returned", PROCESSOR_MAX_POOL_SIZE,
                context.getWsClientMaxPoolSize());
    }

    /**
     * <p>
     * Check the value returns by
     * {@link SoapComponentContext#getWsClientMaxPoolSize()} if parameter
     * 'ws-clients-pool-size-max' is set to not number value.
     * </p>
     * <p>
     * Expected value is its default value: max size of the processor pool.
     * </p>
     * 
     * @throws MessagingException
     */
    public void testGetWsClientMaxPoolSize_102() throws MessagingException {

        final Component componentCfg = new Component();

        final Runtimepositivestrictint acceptorPoolSize = new Runtimepositivestrictint();
        acceptorPoolSize.setValue(ACCEPTOR_POOL_SIZE);
        componentCfg.setAcceptorPoolSize(acceptorPoolSize);

        final Runtimepositivestrictint processorMaxPoolSize = new Runtimepositivestrictint();
        processorMaxPoolSize.setValue(PROCESSOR_MAX_POOL_SIZE);
        componentCfg.setProcessorMaxPoolSize(processorMaxPoolSize);

        final Runtimepositivestrictint processorCorePoolSize = new Runtimepositivestrictint();
        processorCorePoolSize.setValue(PROCESSOR_CORE_POOL_SIZE);
        componentCfg.setProcessorPoolSize(processorCorePoolSize);

        final DocumentBuilder docBuilder = DocumentBuilders.takeDocumentBuilder();
        final Document doc = docBuilder.newDocument();
        final Element wsClientPoolSizeMaxElt = doc.createElementNS(
                "http://petals.ow2.org/components/soap/version-4", "ws-clients-pool-size-max");
        wsClientPoolSizeMaxElt.setTextContent("abcdef");
        componentCfg.getAny().add(wsClientPoolSizeMaxElt);

        final SoapComponentContext context = new SoapComponentContext(componentCfg,
                this.monitoringMbean.getOutgoingProbes(), LOG);

        TestCase.assertEquals("Default value is not returned", PROCESSOR_MAX_POOL_SIZE,
                context.getWsClientMaxPoolSize());
    }

    /**
     * <p>
     * Check the value returns by
     * {@link SoapComponentContext#getWsClientMaxPoolSize()} if parameter
     * 'ws-clients-pool-size-max' is set to an empty {@link String}.
     * </p>
     * <p>
     * Expected value is its default value: max size of the processor pool.
     * </p>
     * 
     * @throws MessagingException
     */
    public void testGetWsClientMaxPoolSize_103() throws MessagingException {

        final Component componentCfg = new Component();

        final Runtimepositivestrictint acceptorPoolSize = new Runtimepositivestrictint();
        acceptorPoolSize.setValue(ACCEPTOR_POOL_SIZE);
        componentCfg.setAcceptorPoolSize(acceptorPoolSize);

        final Runtimepositivestrictint processorMaxPoolSize = new Runtimepositivestrictint();
        processorMaxPoolSize.setValue(PROCESSOR_MAX_POOL_SIZE);
        componentCfg.setProcessorMaxPoolSize(processorMaxPoolSize);

        final Runtimepositivestrictint processorCorePoolSize = new Runtimepositivestrictint();
        processorCorePoolSize.setValue(PROCESSOR_CORE_POOL_SIZE);
        componentCfg.setProcessorPoolSize(processorCorePoolSize);

        final DocumentBuilder docBuilder = DocumentBuilders.takeDocumentBuilder();
        final Document doc = docBuilder.newDocument();
        final Element wsClientPoolSizeMaxElt = doc.createElementNS(
                "http://petals.ow2.org/components/soap/version-4", "ws-clients-pool-size-max");
        wsClientPoolSizeMaxElt.setTextContent("");
        componentCfg.getAny().add(wsClientPoolSizeMaxElt);

        final SoapComponentContext context = new SoapComponentContext(componentCfg,
                this.monitoringMbean.getOutgoingProbes(), LOG);

        TestCase.assertEquals("Default value is not returned", PROCESSOR_MAX_POOL_SIZE,
                context.getWsClientMaxPoolSize());
    }

    // TODO: Add test about unset parameters (a null value for each parameter should use the default value) 

}
