/**
 * PETALS - PETALS Services Platform. Copyright (c) 2008 EBM Websourcing,
 * http://www.ebmwebsourcing.com/
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version. This library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * -------------------------------------------------------------------------
 * $Id$
 * -------------------------------------------------------------------------
 */

package org.ow2.petals.component.framework.api.configuration;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.ow2.petals.component.framework.api.exception.PEtALSCDKException;
import org.ow2.petals.component.framework.api.util.PropertiesUtil;
import org.ow2.petals.component.framework.jbidescriptor.generated.Param;
import org.w3c.dom.Element;

/**
 * The CDK component specific extended configuration. An instance of this object
 * is build with a List of DOM elements. The DOM elements correspond to the
 * defined elements in the CDK component specific XML schema. <br>
 * The parameter are stored in a key/value form.
 * 
 * TODO : Refactor this class in a more elegant way
 * 
 * @author rnaudin - EBM WebSourcing
 * @since 1.0
 */
public class ConfigurationExtensions implements Map<String, String> {

    /**
     * The Map of extended configuration elements.
     */
    private final Map<String, String> extensions;

    /**
     * The properties storing Place Holders values.
     */
    private Properties properties;

    /**
     * 
     * @param configurationElements
     * @param propertiesFilePath
     * @throws PEtALSCDKException
     */
    public ConfigurationExtensions(final List<?> configurationElements, final String propertiesFilePath)
    throws PEtALSCDKException {
        this(configurationElements);

        if (propertiesFilePath != null) {
            this.properties = PropertiesUtil.loadProperties(propertiesFilePath);

            PropertiesUtil.fillPlaceHolder(this.properties, this.extensions);
        }
    }

    /**
     * 
     * @param configurationElements
     */
    public ConfigurationExtensions(final List<?> configurationElements) {
        this.extensions = new ConcurrentHashMap<String, String>();

        if (configurationElements != null && !configurationElements.isEmpty()) {

            final List<ConfigurationElementCardinality> cardinalitiesList = getMultipleElements(configurationElements);

            for (final Object configurationElement : configurationElements) {
                boolean isFound = false;
                for (final ConfigurationElementCardinality elementCardinality : cardinalitiesList) {
                    if (elementCardinality.name.equals(getCfgEltName(configurationElement))) {
                        elementCardinality.suffixNumber++;
                        this.extensions.put(elementCardinality.name
                                + Integer.valueOf(elementCardinality.suffixNumber),
                                getCfgEltValue(configurationElement));
                        isFound = true;
                    }
                }
                if (!isFound) {
                    this.extensions.put(getCfgEltName(configurationElement),
                            getCfgEltValue(configurationElement));
                }
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#clear()
     */
    public void clear() {
        this.extensions.clear();

    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#containsKey(java.lang.Object)
     */
    public boolean containsKey(final Object key) {
        return this.extensions.containsKey(key);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#containsValue(java.lang.Object)
     */
    public boolean containsValue(final Object value) {
        return this.extensions.containsValue(value);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#entrySet()
     */
    public Set<Entry<String, String>> entrySet() {
        return this.extensions.entrySet();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#get(java.lang.Object)
     */
    public String get(final Object key) {
        return this.extensions.get(key);
    }

    /**
     * 
     * @param key
     * @param defaultValue
     * @return
     */
    public String get(final Object key, final String defaultValue) {
        String result = this.get(key);
        if (result == null) {
            result = defaultValue;
        }
        return result;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#isEmpty()
     */
    public boolean isEmpty() {
        return this.extensions.isEmpty();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#keySet()
     */
    public Set<String> keySet() {
        return this.extensions.keySet();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#put(java.lang.Object, java.lang.Object)
     */
    public String put(final String key, final String value) {
        if (this.properties != null) {
            PropertiesUtil.fillPlaceHolder(value, PropertiesUtil.extractPlaceHolderKeys(key),
                    this.properties);
        }
        return this.extensions.put(key, value);
    }

    /**
     * (non-Javadoc)
     * 
     * @see java.util.Map#putAll(java.util.Map)
     */
    @SuppressWarnings("unchecked")
    public void putAll(final Map<? extends String, ? extends String> map) {
        if (this.properties != null) {
            PropertiesUtil.fillPlaceHolder(this.properties, (Map<String, String>) map);
        }
        this.extensions.putAll(map);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#remove(java.lang.Object)
     */
    public String remove(final Object key) {
        return this.extensions.remove(key);
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#size()
     */
    public int size() {
        return this.extensions.size();
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.util.Map#values()
     */
    public Collection<String> values() {
        return this.extensions.values();
    }

    final private static String getCfgEltName(final Object configurationElement) {
        String sRes = null;
        if (configurationElement instanceof Element) {
            sRes = ((Element)configurationElement).getLocalName();
        } else if (configurationElement instanceof Param) {
            sRes = ((Param)configurationElement).getName();
        }

        return sRes;
    }

    final private static String getCfgEltValue(final Object configurationElement) {
        String sRes = null;
        if (configurationElement instanceof Element) {
            sRes = ((Element)configurationElement).getTextContent();
        } else if (configurationElement instanceof Param) {
            sRes = ((Param)configurationElement).getValue();
        }

        return sRes;
    }

    final static private List<ConfigurationElementCardinality> getMultipleElements(final List<?> configurationElements) {
        final List<ConfigurationElementCardinality> cardinalitiesList = new ArrayList<ConfigurationElementCardinality>();

        for (final Object configurationElement : (List<?>) configurationElements) {
            boolean isFound = false;
            for (final ConfigurationElementCardinality elementCardinality : cardinalitiesList) {
                if (elementCardinality.name.equals(getCfgEltName(configurationElement))) {
                    elementCardinality.cardinality++;
                    isFound = true;
                    break;
                }
            }
            if (!isFound) {
                final ConfigurationElementCardinality elementCardinality = new ConfigurationElementCardinality();
                elementCardinality.name = getCfgEltName(configurationElement);
                cardinalitiesList.add(elementCardinality);
            }
        }

        final List<ConfigurationElementCardinality> cardinalitiesList2 = new ArrayList<ConfigurationElementCardinality>();
        for (final ConfigurationElementCardinality elementCardinality : cardinalitiesList) {
            if (elementCardinality.cardinality > 1) {
                cardinalitiesList2.add(elementCardinality);
            }
        }

        return cardinalitiesList2;
    }

    @Override
    public String toString() {
        return this.extensions.toString();
    }

}
